<?php

namespace App\Http\Controllers;

use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Tymon\JWTAuth\Facades\JWTAuth;

class AdminController extends Controller
{
    // REGISTER (optional – you can disable later)
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:admins,email',
            'mobile_number' => 'required|unique:admins,mobile_number',
            'password' => 'required|min:4',
        ]);

        $admin = Admin::create([
            'name' => $request->name,
            'email' => $request->email,
            'mobile_number' => $request->mobile_number,
            'password' => Hash::make($request->password),
            'role' => 'admin',
            'status' => 'active'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Admin created',
            'admin' => $admin
        ]);
    }

    // LOGIN
    public function login(Request $request)
    {
        $request->validate([
            'login' => 'required',
            'password' => 'required'
        ]);

        $admin = Admin::where('email', $request->login)
            ->orWhere('mobile_number', $request->login)
            ->first();

        if (!$admin || !Hash::check($request->password, $admin->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        if ($admin->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Admin blocked'
            ], 403);
        }

        $token = JWTAuth::fromUser($admin);

        return response()->json([
            'success' => true,
            'token' => $token,
            'token_type' => 'bearer',
            'expires_in' => JWTAuth::factory()->getTTL() * 60,
            'admin' => $admin
        ]);
    }

    // PROFILE
    public function profile()
    {
        return response()->json([
            'success' => true,
            'admin' => auth('admin')->user()
        ]);
    }

    // LOGOUT
    public function logout()
    {
        JWTAuth::invalidate(JWTAuth::getToken());

        return response()->json([
            'success' => true,
            'message' => 'Logged out'
        ]);
    }

    // REFRESH
    public function refresh()
    {
        return response()->json([
            'success' => true,
            'token' => JWTAuth::refresh(JWTAuth::getToken())
        ]);
    }
}
