<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\Rider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RiderController extends Controller
{
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:riders,email',
            'mobile_number' => 'required|unique:riders,mobile_number',
            'password' => 'required|min:6',

            'door_no' => 'required',
            'street_name' => 'required',
            'area' => 'required',
            'city' => 'required',
            'state_id' => 'required|exists:states,id',
            'district_id' => 'required|exists:districts,id',
            'pincode' => 'required',
            'country' => 'required',

            'vehicle_type' => 'required',
            'vehicle_number' => 'required|unique:riders,vehicle_number',
            'driving_license_no' => 'required|unique:riders,driving_license_no',
            'rc_book_no' => 'required|unique:riders,rc_book_no',

            'aadhar_number' => 'required|digits:12|unique:riders,aadhar_number',
            'pan_number' => 'required|unique:riders,pan_number',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $otp = '1111'; // TEMP
        $token = Str::uuid()->toString();

        Cache::put(
            'rider_register_' . $token,
            [
                'otp' => $otp,
                'otp_created_at' => now(),
                'data' => $request->all()
            ],
            now()->addMinutes(10)
        );

        return response()->json([
            'success' => true,
            'message' => 'OTP generated. Please verify.',
            'registration_token' => $token,
            'otp' => $otp // TEMP (remove in prod)
        ]);
    }

    public function verifyOtp(Request $request)
    {
        $request->validate([
            'registration_token' => 'required',
            'otp' => 'required'
        ]);

        $cacheKey = 'rider_register_' . $request->registration_token;
        $cached = Cache::get($cacheKey);

        if (!$cached) {
            return response()->json([
                'success' => false,
                'message' => 'Session expired. Please register again.'
            ], 400);
        }

        if (
            $cached['otp'] !== $request->otp ||
            now()->diffInMinutes($cached['otp_created_at']) > 5
        ) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired OTP'
            ], 400);
        }

        $data = $cached['data'];

        $rider = Rider::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'mobile_number' => $data['mobile_number'],
            'password' => Hash::make($data['password']),

            'door_no' => $data['door_no'],
            'street_name' => $data['street_name'],
            'area' => $data['area'],
            'city' => $data['city'],
            'state_id' => $data['state_id'],
            'district_id' => $data['district_id'],
            'pincode' => $data['pincode'],
            'country' => $data['country'],

            'vehicle_type' => $data['vehicle_type'],
            'vehicle_number' => $data['vehicle_number'],
            'driving_license_no' => $data['driving_license_no'],
            'rc_book_no' => $data['rc_book_no'],

            'pan_number' => $data['pan_number'],
            'aadhar_number' => $data['aadhar_number'],

            'status' => 'pending_admin',

            'admin_remark' => null,
            'approved_at' => null,
            'approved_by' => null,
        ]);

        Address::create([
            'user_type'     => 'rider',
            'user_id'       => $rider->id,
            'name'          => $data['name'],
            'mobile_number' => $data['mobile_number'],
            'door_no'       => $data['door_no'],
            'street_name'   => $data['street_name'],
            'area'          => $data['area'],
            'city'          => $data['city'],
            'district_id'   => $data['district_id'],
            'state_id'      => $data['state_id'],
            'pincode'       => $data['pincode'],
            'is_default'    => 1
        ]);

        Cache::forget($cacheKey);

        return response()->json([
            'success' => true,
            'message' => 'OTP verified. Rider registration completed.',
            'rider_id' => $rider->id
        ]);
    }


    public function login(Request $request)
    {
        $request->validate([
            'login'=>'required',
            'password'=>'required'
        ]);

        $rider = Rider::where('email',$request->login)
            ->orWhere('mobile_number',$request->login)
            ->first();

        if (!$rider || !Hash::check($request->password,$rider->password)) {
            return response()->json(['message'=>'Invalid credentials'],401);
        }

        if ($rider->status !== 'active') {
            return response()->json(['message'=>'Account blocked'],403);
        }

        $token = JWTAuth::customClaims([
            'exp' => now()->addDays(30)->timestamp
        ])->fromUser($rider);

        return response()->json([
            'success'=>true,
            'token'=>$token,
            'rider'=>$rider
        ]);
    }

    // ================= PROFILE =================
    public function profile()
    {
        return response()->json([
            'rider' => auth('rider')->user()
        ]);
    }

    public function forgotPassword(Request $request)
    {
        $request->validate(['mobile_number'=>'required']);

        $rider = Rider::where('mobile_number',$request->mobile_number)->first();

        if (!$rider) {
            return response()->json(['message'=>'Not registered'],404);
        }

        $rider->update([
            'otp'=>'1111',
            'otp_created_at'=>now()
        ]);

        return response()->json(['message'=>'OTP sent (1111)']);
    }

    public function verifyForgotOtp(Request $request)
    {
        $rider = Rider::where('otp',$request->otp)->first();

        if (!$rider || now()->diffInMinutes($rider->otp_created_at) > 5) {
            return response()->json(['message'=>'Invalid OTP'],400);
        }

        return response()->json([
            'success'=>true,
            'rider_id'=>$rider->id
        ]);
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'rider_id'=>'required',
            'password'=>'required|confirmed|min:8'
        ]);

        $rider = Rider::find($request->rider_id);

        if (!$rider) {
            return response()->json(['message'=>'Not found'],404);
        }

        $rider->update([
            'password'=>Hash::make($request->password),
            'otp'=>null,
            'otp_created_at'=>null
        ]);

        return response()->json(['message'=>'Password reset successful']);
    }

    public function logout()
    {
        JWTAuth::invalidate(JWTAuth::getToken());
        return response()->json(['message' => 'Logged out']);
    }

    // ================= REFRESH =================
    public function refresh()
    {
        return response()->json([
            'token' => JWTAuth::refresh(JWTAuth::getToken())
        ]);
    }

    public function changePassword(Request $request)
    {
        $rider = auth('rider')->user();

        $request->validate([
            'current_password'      => 'required|string',
            'new_password'          => 'required|string|min:8|confirmed',
            // 'confirmed' expects a field named new_password_confirmation
        ]);

        if (!Hash::check($request->current_password, $rider->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect.'
            ], 400);
        }

        $rider->password = Hash::make($request->new_password);
        $rider->save();

        JWTAuth::invalidate(JWTAuth::getToken());

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully. You have been logged out.'
        ]);
    }

    public function accountDetails()
    {
        $rider = auth('rider')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'name'     => $rider->name,
                'mobile_number'  => $rider->mobile_number,

                'logo_url' => $rider->logo
                    ? url($rider->logo)
                    : null,

                'logo_initial' => strtoupper(
                    substr($rider->rider ?? $rider->name, 0, 1)
                ),
            ]
        ]);
    }

    public function updateLogo(Request $request)
    {
        $rider = auth('rider')->user();

        $request->validate([
            'logo' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);

        if ($rider->logo && file_exists(public_path($rider->logo))) {
            unlink(public_path($rider->logo));
        }

        $file = $request->file('logo');
        $filename = 'rider_' . $rider->id . '_' . time() . '.' . $file->getClientOriginalExtension();

        $uploadPath = 'uploads/rider/logos';
        $file->move(public_path($uploadPath), $filename);

        $rider->update([
            'logo' => $uploadPath . '/' . $filename
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Logo updated successfully',
            'logo_url' => url($uploadPath . '/' . $filename)
        ]);
    }

    public function rateDetails()
    {
        $rider = auth('rider')->user();

        return response()->json([
            'success' => true,
            'rate_change_status' => $rider->rate_change_status,
            'data' => [
                'range'             => $rider->range,
                'rate_per_km'       => $rider->ratePerKm,
            ]
        ]);
    }

    public function updateRateDetails(Request $request)
    {
        $rider = auth('rider')->user();

        $request->validate([
            'range' => 'required|numeric|min:0',
            'rate_per_km' => 'required|numeric|min:0'
        ]);

        $rider->update([
            'range' => $request->range,
            'ratePerKm' => $request->rate_per_km,
            'rate_change_status' => 'pending',
            'rate_approved_at' => null,
            'rate_approved_by' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Rate update request sent for admin approval',
            'data' => [
                'range' => $rider->range,
                'rate_per_km' => $rider->ratePerKm,
                'status' => $rider->rate_change_status
            ]
        ]);
    }

    public function Personaletails(Request $request)
    {
        $rider = auth('rider')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'name'    => $rider->name,
                'mobile_number' => $rider->mobile_number,
                'email'         => $rider->email,
            ]
        ]);
    }

    public function addressDetails()
    {
        $rider = auth('rider')->user()->load(['state', 'district']);

        return response()->json([
            'success' => true,
            'data' => [
                'door_no'         => $rider->door_no,
                'street_name'     => $rider->street_name,
                'area'            => $rider->area,
                'city'            => $rider->city,
                'district_name'   => $rider->district->name ?? null,
                'state_name'      => $rider->state->name ?? null,
                'pincode'         => $rider->pincode,
            ]
        ]);
    }

    public function legalDetails()
    {
        $rider = auth('rider')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'vehicle_type'         => $rider->vehicle_type,
                'vehicle_number'     => $rider->vehicle_number,
                'driving_license_no'            => $rider->driving_license_no,
                'rc_book_no'            => $rider->rc_book_no,
                'aadhar_number'   => $rider->aadhar_number,
                'pan_number'      => $rider->pan_number,
            ]
        ]);
    }

    public function bankDetails()
    {
        $rider = auth('rider')->user()->load('bank');

        return response()->json([
            'success' => true,
            'bank_status' => $rider->bank_status,
            'has_pending_update' => !empty($store->pending_bank_data),

            'data' => [
                'bank_holder_name'    => $rider->bank_holder_name,
                'bank_id'             => $rider->bank_id,
                'bank_name'           => $rider->bank->name ?? null,
                'bank_account_number' => $rider->bank_account_number,
                'ifsc_code'           => $rider->ifsc_code,
                'micr_no'             => $rider->micr_no,
                'upi_id'              => $rider->upi_id,
                'passbook_file'       => $rider->passbook_file 
                    ? asset('storage/' . $rider->passbook_file) 
                    : null,
            ],

            'pending_data' => $rider->pending_bank_data,
            'admin_remark' => $rider->admin_remark,
        ]);
    }


    public function saveBankDetails(Request $request)
    {
        $rider = auth('rider')->user();

        if ($rider->bank_id) {
            return response()->json([
                'success' => false,
                'message' => 'Bank details already exist. Please update instead.'
            ], 403);
        }

        $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'bank_holder_name' => 'required|string|max:255',
            'bank_account_number' => 'required|confirmed',
            'bank_account_number_confirmation' => 'required',
            'ifsc_code' => 'required|string|max:255',
            'micr_no' => 'nullable|string|max:50',
            'upi_id' => 'nullable|string|max:100',
            'passbook_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024',
        ]);

        $file = $request->file('passbook_file');
        $filename = 'passbook_' . $rider->id . '_' . time() . '.' . $file->getClientOriginalExtension();
        $uploadPath = 'rider/passbooks';
        $file->move(public_path($uploadPath), $filename);

        $rider->update([
            'pending_bank_data' => [
                'bank_id' => $request->bank_id,
                'bank_holder_name' => $request->bank_holder_name,
                'bank_account_number' => $request->bank_account_number,
                'ifsc_code' => $request->ifsc_code,
                'micr_no' => $request->micr_no,
                'upi_id' => $request->upi_id,
                'passbook_file' => $uploadPath . '/' . $filename,
            ],
            'bank_status' => 'pending',
            'admin_remark' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Bank details submitted for admin approval'
        ]);
    }

    public function updateBankDetails(Request $request)
    {
        $rider = auth('rider')->user();

        if ($rider->bank_status === 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Already waiting for admin approval'
            ], 403);
        }

        $request->validate([
            'bank_id' => 'sometimes|exists:banks,id',
            'bank_holder_name' => 'sometimes|string|max:255',
            'bank_account_number' => 'sometimes|confirmed',
            'bank_account_number_confirmation' => 'required_with:bank_account_number',
            'ifsc_code' => 'sometimes|string|max:255',
            'micr_no' => 'nullable|string|max:50',
            'upi_id' => 'nullable|string|max:100',
            'passbook_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024',
        ]);

        $pending = $store->pending_bank_data ?? [];

        if ($request->hasFile('passbook_file')) {
            if (!empty($pending['passbook_file']) && file_exists(public_path($pending['passbook_file']))) {
                unlink(public_path($pending['passbook_file']));
            }

            $file = $request->file('passbook_file');
            $filename = 'passbook_' . $rider->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'rider/passbooks';
            $file->move(public_path($uploadPath), $filename);

            $pending['passbook_file'] = $uploadPath . '/' . $filename;
        }

        $rider->update([
            'pending_bank_data' => [
                'bank_id' => $request->bank_id ?? $rider->bank_id,
                'bank_holder_name' => $request->bank_holder_name ?? $rider->bank_holder_name,
                'bank_account_number' => $request->bank_account_number ?? $rider->bank_account_number,
                'ifsc_code' => $request->ifsc_code ?? $rider->ifsc_code,
                'micr_no' => $request->micr_no ?? $rider->micr_no,
                'upi_id' => $request->upi_id ?? $rider->upi_id,
                'passbook_file' => $pending['passbook_file'] ?? $rider->passbook_file,
            ],
            'bank_status' => 'pending',
            'admin_remark' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Updated bank details sent for admin approval'
        ]);
    }

    public function notifications(Request $request)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $notifications = DB::table('notifications')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('created_at', '>=', Carbon::now()->subDays(15))
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function($n){
                $link = $n->link ? json_decode($n->link, true) : null;

                return [
                    'id'         => $n->id,
                    'title'      => $n->title,
                    'content'    => $n->content,
                    'route'      => $link['route'] ?? null,
                    'route_id'   => $link['route_id'] ?? null,
                    'status'     => $n->status,
                    'created_date'=> Carbon::parse($n->created_at)->format('Y-m-d'),
                    'time'       => Carbon::parse($n->created_at)->format('H:i:s'),
                    'time_ago'   => Carbon::parse($n->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'count'   => $notifications->count(),
            'data'    => $notifications
        ]);
    }

    public function activeNotificationsCount(Request $request)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $activeCount = DB::table('notifications')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('status', 'active')
            ->count();

        return response()->json([
            'success' => true,
            'active_count' => $activeCount
        ]);
    }

    public function toggleNotificationStatus(Request $request, $notificationId)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $notification = DB::table('notifications')
            ->where('id', $notificationId)
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->first();

        if ($notification->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Notification is already inactive'
            ], 400);
        }

        DB::table('notifications')
            ->where('id', $notificationId)
            ->update([
                'status'     => 'inactive',
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Notification deactivated successfully',
            'data' => [
                'notification_id' => $notificationId,
                'status' => 'inactive',
            ]
        ], 200);

    }
}
