<?php

namespace App\Http\Controllers;
use App\Events\RiderLocationUpdated;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\OrderDeliveredMail;
use App\Models\Order;
use App\Models\Customer;
use App\Models\Store;

class RiderOrderController extends Controller
{
    private function sendNotification($title, $content, $userType, $userId, $route = null, $routeId = null)
    {
        $linkData = null;

        if ($route) {
            $linkData = $routeId !== null
                ? json_encode(['route' => $route, 'route_id' => $routeId], JSON_UNESCAPED_SLASHES)
                : json_encode(['route' => $route], JSON_UNESCAPED_SLASHES);
        }


        DB::table('notifications')->insert([
            'title'      => $title,
            'content'    => $content,
            'user_type'  => $userType,
            'user_id'    => $userId,
            'status'     => 'active',
            'link'       => $linkData,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        /* ========== 2. Get active FCM tokens ========== */
        $tokens = DB::table('fcm_tokens')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('notify_status', 'active')
            ->pluck('fcm_token')
            ->toArray();

        if (empty($tokens)) {
            return; 
        }

        /* ========== 3. Firebase HTTP v1 Auth ========== */
        $credentialsPath = base_path(env('FIREBASE_CREDENTIALS'));
        $projectId       = env('FIREBASE_PROJECT_ID');

        $client = new \Google_Client();
        $client->setAuthConfig($credentialsPath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');

        $accessToken = $client->fetchAccessTokenWithAssertion();

        if (!isset($accessToken['access_token'])) {
            \Log::error('Firebase token error', $accessToken);
            return;
        }

        $bearerToken = $accessToken['access_token'];

        $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

        foreach ($tokens as $fcmToken) {

            $dataPayload = [
                "route"     => (string) $route,
                "type"      => "notification",
                "user_type" => (string) $userType,
                "status"    => "active",
            ];

            if ($routeId !== null) {
                $dataPayload["route_id"] = (string) $routeId;
            }

            $payload = [
                "message" => [
                    "token" => $fcmToken,
                    "notification" => [
                        "title" => $title,
                        "body"  => $content,
                    ],
                    "data" => $dataPayload,
                    "android" => [
                        "priority" => "high",
                        "notification" => [
                            "sound" => "default",
                            "click_action" => "FLUTTER_NOTIFICATION_CLICK"
                        ]
                    ]
                ]
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Authorization: Bearer {$bearerToken}",
                "Content-Type: application/json"
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

            $response = curl_exec($ch);

            if ($response === false) {
                \Log::error('Firebase Push Error: ' . curl_error($ch));
            }

            curl_close($ch);
        }
    }

    public function updateRiderLocation(Request $request)
    {
        $rider = auth('rider')->user();

        if ($rider->is_available != 1) {
            return response()->json([
                'success' => false,
                'message' => 'Rider is offline. Location updates not allowed.'
            ], 403);
        }

        $request->validate([
            'latitude'  => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        DB::table('rider_locations')->updateOrInsert(
            ['rider_id' => $rider->id],
            [
                'latitude'   => $request->latitude,
                'longitude'  => $request->longitude,
                'updated_at' => now(),
            ]
        );

        event(new RiderLocationUpdated(
            $rider->id,
            $request->latitude,
            $request->longitude
        ));


        return response()->json([
            'success' => true,
            'message' => 'Location updated & broadcasted',
            'data' => [
                'rider_id' => $rider->id,
                'latitude' => $request->latitude,
                'longitude'=> $request->longitude,
            ]
        ]);
    }

    public function toggleAvailability(Request $request)
    {
        $rider = auth('rider')->user();

        $request->validate([
            'is_available' => 'required|in:0,1'
        ]);

        if ($request->is_available == 0) {

            $hasActiveOrder = DB::table('orders')
                ->where('rider_id', $rider->id)
                ->whereIn('status', [
                    'rider_accepted',
                    'rider_verified',
                    'in_transit'
                ])
                ->exists();

            if ($hasActiveOrder) {
                return response()->json([
                    'success' => false,
                    'message' => 'You cannot go offline while you have an active order'
                ], 400);
            }

            $hasActiveRequest = DB::table('order_rider_requests')
                ->where('rider_id', $rider->id)
                ->where('status', 'active')
                ->exists();

            if ($hasActiveRequest) {
                return response()->json([
                    'success' => false,
                    'message' => 'You cannot go offline while you have an active request'
                ], 400);
            }
        }

        DB::table('riders')
            ->where('id', $rider->id)
            ->update([
                'is_available' => $request->is_available,
                'updated_at'   => now()
            ]);

        return response()->json([
            'success' => true,
            'message' => $request->is_available
                ? 'You are now available'
                : 'You are now unavailable',
            'is_available' => (int) $request->is_available
        ]);
    }


    public function riderRequestedOrders()
    {
        $rider = auth('rider')->user();

        $riderData = DB::table('riders')
            ->where('id', $rider->id)
            ->first();

        $workStatus = (bool) $riderData->work_status; 

        $orders = DB::table('order_rider_requests as orr')
            ->join('orders as o', 'o.id', '=', 'orr.order_id')

            ->leftJoin('warehouses as w', function ($join) {
                $join->on('w.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'warehouse');
            })
            ->leftJoin('stores as s', function ($join) {
                $join->on('s.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'store');
            })

            ->leftJoin(
                'districts as pd',
                DB::raw('pd.id'),
                '=',
                DB::raw('COALESCE(w.district_id, s.district_id)')
            )
            ->leftJoin(
                'states as ps',
                DB::raw('ps.id'),
                '=',
                DB::raw('COALESCE(w.state_id, s.state_id)')
            )

            ->leftJoin('addresses as a', 'a.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'a.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'a.state_id')

            ->where('orr.rider_id', $rider->id)
            ->where('orr.status', 'requested')
            ->where('o.status', 'waiting_for_rider_approval')

            ->orderBy('orr.requested_at', 'desc')

            ->select(
                'orr.id as request_id',
                'orr.distance_km',
                'orr.requested_at',

                'o.id as order_id',
                'o.order_no',
                'o.created_at as order_created_at',

                DB::raw("COALESCE(w.warehouse_name, s.store_name) as pickup_name"),
                DB::raw("COALESCE(w.door_no, s.door_no) as pickup_door_no"),
                DB::raw("COALESCE(w.street_name, s.street_name) as pickup_street"),
                DB::raw("COALESCE(w.area, s.area) as pickup_area"),
                DB::raw("COALESCE(w.city, s.city) as pickup_city"),
                DB::raw("COALESCE(w.pincode, s.pincode) as pickup_pincode"),
                'pd.name as pickup_district',
                'ps.name as pickup_state',
                
                'a.name as drop_name',
                'a.door_no as drop_door_no',
                'a.street_name as drop_street',
                'a.area as drop_area',
                'a.city as drop_city',
                'a.pincode as drop_pincode',
                'dd.name as drop_district',
                'ds.name as drop_state'
            )
            ->get()
            ->map(function ($row) use ($rider) {

                $materials = DB::table('order_items as oi')
                    ->join('products as p', 'p.id', '=', 'oi.product_id')
                    ->where('oi.order_id', $row->order_id)
                    ->select(
                        'p.category',
                        'p.subcategory',
                        'p.brand',
                        'oi.quantity',
                        'p.quantity_unit'
                    )
                    ->get();

                $distance = (float) $row->distance_km;
                $ratePerKm = (float) $rider->ratePerKm;
                $tripAmount = $distance * $ratePerKm;

                return [
                    'request_id' => $row->request_id,
                    'order_id'   => $row->order_id, 
                    'order_no'   => $row->order_no,

                    'materials' => $materials,

                    'pickup_location' => [
                        'name'     => $row->pickup_name,
                        'door_no'  => $row->pickup_door_no,
                        'street'   => $row->pickup_street,
                        'area'     => $row->pickup_area,
                        'city'     => $row->pickup_city,
                        'district' => $row->pickup_district,
                        'state'    => $row->pickup_state,
                        'pincode'  => $row->pickup_pincode,
                    ],

                    'drop_location' => [
                        'name'     => $row->drop_name,
                        'door_no'  => $row->drop_door_no,
                        'street'   => $row->drop_street,
                        'area'     => $row->drop_area,
                        'city'     => $row->drop_city,
                        'district' => $row->drop_district,
                        'state'    => $row->drop_state,
                        'pincode'  => $row->drop_pincode,
                    ],

                    'delivery_date' => Carbon::parse($row->order_created_at)
                        ->addDays(7)
                        ->format('d/m/Y'),

                    'distance_km' => $distance,
                    'rate_per_km' => $ratePerKm,
                    'trip_amount' => $tripAmount,

                    'requested_date' => Carbon::parse($row->requested_at)->format('d/m/Y'),
                    'requested_time' => Carbon::parse($row->requested_at)->format('h:i A'),
                ];
            });

        return response()->json([
            'success' => true,
            'rider_work_status' => $workStatus,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function acceptOrder(Request $request, $orderId)
    {
        $rider = auth('rider')->user();

        DB::beginTransaction();
        try {

            $riderData = DB::table('riders')
                ->where('id', $rider->id)
                ->lockForUpdate()
                ->first();

            if ($riderData->status !== 'active') {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Your account is not active'
                ], 403);
            }

            if ($riderData->work_status == 1) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'You are already busy with another order'
                ], 409);
            }

            if ($riderData->is_available != 1) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'You are not available to accept orders'
                ], 409);
            }

            $order = DB::table('orders')
                ->where('id', $orderId)
                ->where('status', 'waiting_for_rider_approval')
                ->lockForUpdate()
                ->first();

            if (!$order) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Order not available for acceptance'
                ], 404);
            }

            if ($order->rider_id !== null) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Order already accepted by another rider'
                ], 409);
            }

            $riderRequest = DB::table('order_rider_requests')
                ->where('order_id', $orderId)
                ->where('rider_id', $rider->id)
                ->where('status', 'requested')
                ->lockForUpdate()
                ->first();

            $distance = (float) $riderRequest->distance_km;
            $ratePerKm = (float) $rider->ratePerKm;
            $tripAmount = round($distance * $ratePerKm, 2);

            if (!$riderRequest) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'No active request found for this order'
                ], 403);
            }

            $otp = rand(1000, 9999);

            DB::table('order_rider_requests')
                ->where('id', $riderRequest->id)
                ->update([
                    'status'       => 'active',
                    'trip_amount'  => $tripAmount,
                    'otp'          => $otp,
                    'accepted_at'  => now(),
                    'responded_at' => now(),
                    'updated_at'   => now()
                ]);

            DB::table('orders')
                ->where('id', $orderId)
                ->update([
                    'rider_id'   => $rider->id,
                    'status'     => 'rider_accepted',
                    'updated_at' => now()
                ]);

            DB::table('order_rider_requests')
                ->where('order_id', $orderId)
                ->where('id', '!=', $riderRequest->id)
                ->where('status', 'requested')
                ->update([
                    'status'       => 'expired',
                    'responded_at' => now(),
                    'updated_at'   => now()
                ]);


            DB::table('riders')
                ->where('id', $rider->id)
                ->update([
                    'work_status' => 1
                ]);

            DB::commit();
        
            if ($order->approved_by && $order->approved_by_user_type) {

                $approveById   = $order->approved_by;
                $approveByType = $order->approved_by_user_type;

                $route = "{$approveByType}/orders/assigned-orders";

                $this->sendNotification(
                    'Rider Accepted Order',
                    "Rider {$rider->name} has accepted Order #{$order->order_no}. Tap to view details.",
                    $approveByType,   // user_type
                    $approveById,     // user_id
                    $route,           // route
                    $orderId          // route_id
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Order accepted successfully',
                'order_id' => $orderId,
                'otp'     => $otp
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function rejectOrder(Request $request, $orderId)
    {
        $rider = auth('rider')->user();

        DB::beginTransaction();

        try {
            $riderData = DB::table('riders')
                ->where('id', $rider->id)
                ->lockForUpdate()
                ->first();

            if (!$riderData || $riderData->status !== 'active') {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Your account is not active'
                ], 403);
            }

            // Rider must be available (toggle ON)
            if ($riderData->is_available != 1) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'You cannot reject orders right now'
                ], 409);
            }

            $updated = DB::table('order_rider_requests')
                ->where('order_id', $orderId)
                ->where('rider_id', $rider->id)
                ->where('status', 'requested')
                ->update([
                    'status'       => 'rejected',
                    'responded_at' => now(),
                    'updated_at'   => now()
                ]);

            if (!$updated) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Request not found or already handled'
                ], 404);
            }

            DB::table('riders')
                ->where('id', $rider->id)
                ->update([
                    'is_available' => 1,
                    'updated_at'   => now()
                ]);

            DB::commit();

            return response()->json([
                'success'  => true,
                'message'  => 'Order rejected successfully',
                'order_id' => $orderId
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function riderAcceptedOrders()
    {
        $rider = auth('rider')->user();

        $orders = DB::table('orders as o')
            ->join('order_rider_requests as orr', function ($join) use ($rider) {
                $join->on('orr.order_id', '=', 'o.id')
                    ->where('orr.rider_id', $rider->id)
                    ->where('orr.status', 'active');
            })

            ->leftJoin('warehouses as w', function ($join) {
                $join->on('w.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'warehouse');
            })
            ->leftJoin('stores as s', function ($join) {
                $join->on('s.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'store');
            })

            ->leftJoin(
                'districts as pd',
                DB::raw('pd.id'),
                '=',
                DB::raw('COALESCE(w.district_id, s.district_id)')
            )
            ->leftJoin(
                'states as ps',
                DB::raw('ps.id'),
                '=',
                DB::raw('COALESCE(w.state_id, s.state_id)')
            )

            ->where('o.rider_id', $rider->id)
            ->where('o.status', 'rider_accepted')

            ->orderBy('o.updated_at', 'desc')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.created_at as order_created_at',

                DB::raw("COALESCE(w.warehouse_name, s.store_name) as pickup_name"),
                DB::raw("COALESCE(w.door_no, s.door_no) as pickup_door_no"),
                DB::raw("COALESCE(w.street_name, s.street_name) as pickup_street"),
                DB::raw("COALESCE(w.area, s.area) as pickup_area"),
                DB::raw("COALESCE(w.city, s.city) as pickup_city"),
                DB::raw("COALESCE(w.pincode, s.pincode) as pickup_pincode"),
                'pd.name as pickup_district',
                'ps.name as pickup_state'
            )
            ->get()
            ->map(function ($row) {

                $products = DB::table('order_items as oi')
                    ->join('products as p', 'p.id', '=', 'oi.product_id')
                    ->where('oi.order_id', $row->order_id)
                    ->select(
                        'p.category',
                        'p.subcategory',
                        'p.brand',
                        'oi.quantity',
                        'p.quantity_unit'
                    )
                    ->get();

                return [
                    'order_id' => $row->order_id,
                    'order_no' => $row->order_no,

                    'products' => $products,

                    'pickup_location' => [
                        'name'     => $row->pickup_name,
                        'door_no'  => $row->pickup_door_no,
                        'street'   => $row->pickup_street,
                        'area'     => $row->pickup_area,
                        'city'     => $row->pickup_city,
                        'district' => $row->pickup_district,
                        'state'    => $row->pickup_state,
                        'pincode'  => $row->pickup_pincode,
                    ],

                    'accepted_at' => Carbon::parse($row->order_created_at)
                                            ->format('d/m/Y h:i A'),
                ];
            });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function riderOrderDetail($orderId)
    {
        $rider = auth('rider')->user();

        $order = DB::table('orders as o')
            ->leftJoin('warehouses as w', function ($join) {
                $join->on('w.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'warehouse');
            })
            ->leftJoin('stores as s', function ($join) {
                $join->on('s.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'store');
            })
            ->leftJoin('districts as pd', DB::raw('pd.id'), '=', DB::raw('COALESCE(w.district_id, s.district_id)'))
            ->leftJoin('states as ps', DB::raw('ps.id'), '=', DB::raw('COALESCE(w.state_id, s.state_id)'))
            ->leftJoin('addresses as a', 'a.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'a.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'a.state_id')
            ->leftJoin('order_rider_requests as orr', function ($join) use ($rider) {
                $join->on('orr.order_id', '=', 'o.id')
                    ->where('orr.rider_id', '=', $rider->id)
                    ->where('orr.status', '=', 'active');
            })
            ->where('o.id', $orderId)
            ->where('o.rider_id', $rider->id)
            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'orr.otp',

                DB::raw("COALESCE(w.warehouse_name, s.store_name) as pickup_name"),
                DB::raw("COALESCE(w.door_no, s.door_no) as pickup_door_no"),
                DB::raw("COALESCE(w.street_name, s.street_name) as pickup_street"),
                DB::raw("COALESCE(w.area, s.area) as pickup_area"),
                DB::raw("COALESCE(w.city, s.city) as pickup_city"),
                DB::raw("COALESCE(w.pincode, s.pincode) as pickup_pincode"),
                DB::raw("COALESCE(w.latitude, s.latitude) as pickup_lat"),
                DB::raw("COALESCE(w.longitude, s.longitude) as pickup_lng"),
                DB::raw("COALESCE(w.mobile_number, s.mobile_number) as pickup_phone"),

                'pd.name as pickup_district',
                'ps.name as pickup_state',

                'a.name as drop_name',
                'a.mobile_number as drop_phone',
                'a.door_no as drop_door_no',
                'a.street_name as drop_street',
                'a.area as drop_area',
                'a.city as drop_city',
                'a.pincode as drop_pincode',
                'a.latitude as drop_lat',
                'a.longitude as drop_lng',
                'dd.name as drop_district',
                'ds.name as drop_state'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'p.category',
                'p.subcategory',
                'p.brand',
                'oi.quantity',
                'p.quantity_unit'
            )
            ->get();

        $riderLocation = DB::table('rider_locations')
            ->where('rider_id', $rider->id)
            ->orderBy('created_at', 'desc')
            ->first();


        $tracking = null;

        if ($riderLocation) {
            if ($order->status == 'rider_accepted') {
                $tracking = [
                    'from' => [
                        'lat' => $riderLocation->latitude,
                        'lng' => $riderLocation->longitude,
                    ],
                    'to' => [
                        'lat' => $order->pickup_lat,
                        'lng' => $order->pickup_lng,
                    ],
                    'description' => 'Rider heading to pickup location'
                ];
            } 
            elseif ($order->status == 'in_transit') {
                $tracking = [
                    'from' => [
                        'lat' => $riderLocation->latitude,
                        'lng' => $riderLocation->longitude,
                    ],
                    'to' => [
                        'lat' => $order->drop_lat,
                        'lng' => $order->drop_lng,
                    ],
                    'description' => 'Rider heading to drop location'
                ];
            }
        }


        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'otp'      => $order->otp,
                'products' => $products,

                'pickup_location' => [
                    'name'     => $order->pickup_name,
                    'phone'    => $order->pickup_phone,
                    'door_no'  => $order->pickup_door_no,
                    'street'   => $order->pickup_street,
                    'area'     => $order->pickup_area,
                    'city'     => $order->pickup_city,
                    'district' => $order->pickup_district,
                    'state'    => $order->pickup_state,
                    'pincode'  => $order->pickup_pincode,
                    'latitude' => $order->pickup_lat,
                    'longitude'=> $order->pickup_lng,
                ],

                'drop_location' => [
                    'name'     => $order->drop_name,
                    'phone'    => $order->drop_phone,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                    'latitude' => $order->drop_lat,
                    'longitude'=> $order->drop_lng,
                ],

                'tracking' => $tracking
            ]
        ]);
    }

    public function startRideByRider(Request $request)
    {
        $rider = auth('rider')->user();

        $request->validate([
            'order_id'   => 'required|exists:orders,id',
            'loaded_otp' => 'required|digits:4'
        ]);

        $order = DB::table('orders')
            ->where('id', $request->order_id)
            ->where('rider_id', $rider->id)
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found for this rider'
            ], 404);
        }

        if (!$order->loaded_otp) {
            return response()->json([
                'success' => false,
                'message' => 'Warehouse has not generated loaded OTP yet'
            ], 400);
        }

        if ((string) $order->loaded_otp !== (string) $request->loaded_otp) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid loaded OTP'
            ], 400);
        }

        DB::beginTransaction();

        try {
            $deliveryOtp = $order->delivery_otp ?? rand(1000, 9999);

            $newStatus = $order->with_shipping ? 'in_transit' : 'completed';

            $updateData = [
                'status'          => $newStatus,
                'ride_started_at' => now(),
                'delivery_otp'    => $deliveryOtp,
                'updated_at'      => now()
            ];

            if (!$order->with_shipping) {
                $updateData['delivered_at'] = now();
            }

            DB::table('orders')
                ->where('id', $request->order_id)
                ->update($updateData);

            DB::table('riders')
                ->where('id', $rider->id)
                ->update([
                    'work_status' => 0,
                    'updated_at'  => now()
                ]);

            DB::commit();

            $route = $order->user_type . '/my-purchase'; 

            $this->sendNotification(
                $newStatus === 'completed' ? 'Order Completed' : 'Order In Transit',
                $newStatus === 'completed' 
                    ? 'Your order has been completed successfully.' 
                    : 'Your order has been picked up and is on the way',
                $order->user_type,
                $order->user_id,
                $route,
            );

            return response()->json([
                'success' => true,
                'message' => $order->with_shipping 
                    ? 'Loaded OTP verified, ride started'
                    : 'Order marked as completed (no shipping required)',
                'data' => [
                    'order_id' => $order->id,
                    'status'   => $newStatus
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to start ride',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    private function sendOrderDeliveredEmail($order, $isSeller = false)
    {
        if (!$order) return;

        if ($order instanceof \stdClass) {
            $order = Order::find($order->id);
            if (!$order) return;
        }

        // Load order item
        $order->item = DB::table('order_items as oi')
            ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->where('oi.order_id', $order->id)
            ->select(
                'oi.quantity',
                'oi.total_price',
                'p.brand',
                'p.quantity_unit',
                's.name as subcategory_name',
                DB::raw('CASE WHEN oi.quantity > 0 THEN oi.total_price / oi.quantity ELSE 0 END as unit_price')
            )
            ->first();

        if (!$order->item) return;

        // Identify user
        $user = null;

        if ($isSeller) {
            if ($order->approved_by_user_type === 'warehouse') {
                $user = Warehouse::find($order->approved_by);
            } elseif ($order->approved_by_user_type === 'store') {
                $user = Store::find($order->approved_by);
            }
        } else {
            if ($order->user_type === 'store') {
                $user = Store::find($order->user_id);
            } elseif ($order->user_type === 'customer') {
                $user = Customer::find($order->user_id);
            }
        }

        if (!$user || empty($user->email)) return;

        // ✅ Send via Mailable
        Mail::to($user->email)
            ->send(new OrderDeliveredMail($order, $user, $isSeller));
    }
    
    public function verifyDeliveryOtp(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'otp' => 'required|digits:4'
        ]);

        $order = DB::table('orders')->where('id', $request->order_id)->first();

        if (!$order) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

       if ($order->delivery_otp == $request->otp) {

            DB::table('orders')->where('id', $request->order_id)->update([
                'status' => 'completed',
                'delivered_at' => Carbon::now()
            ]);

            DB::table('order_rider_requests')
                ->where('order_id', $request->order_id)
                ->where('rider_id', $order->rider_id)
                ->update([
                    'status' => 'completed',
                    'responded_at' => Carbon::now()
                ]);

            if ($order->user_type && $order->user_id) {

                $buyerRoute = "{$order->user_type}/my-purchase";  

                $this->sendNotification(
                    'Order Delivered Successfully',
                    "Your order #{$order->order_no} has been delivered successfully.",
                    $order->user_type,
                    $order->user_id,
                    $buyerRoute,
                    $order->id
                );

                $this->sendOrderDeliveredEmail($order, false);
            }

            if ($order->approved_by && $order->approved_by_user_type) {

                $sellerRoute = "{$order->approved_by_user_type}/orders/delivered-orders";

                $this->sendNotification(
                    'Order Delivered',
                    "Order #{$order->order_no} has been successfully delivered to the customer.",
                    $order->approved_by_user_type,
                    $order->approved_by,
                    $sellerRoute,
                    $order->id
                );
                $this->sendOrderDeliveredEmail($order, true);
            }

            return response()->json([
                'success' => true,
                'message' => 'OTP verified, order marked as delivered and rider request completed'
            ]);
        }

        return response()->json(['success' => false, 'message' => 'Invalid OTP'], 400);
    }

    public function riderDeliveredOrders()
    {
        $rider = auth('rider')->user();

        $riderData = DB::table('riders')
            ->where('id', $rider->id)
            ->first();

        // $workStatus = (bool) $riderData->work_status; 

        $orders = DB::table('order_rider_requests as orr')
            ->join('orders as o', 'o.id', '=', 'orr.order_id')
            ->where('orr.rider_id', $rider->id)
            ->where('orr.status', 'completed')
            ->where('o.status', 'completed')
            ->orderBy('o.delivered_at', 'desc') // sort by delivered time
            ->select(
                'orr.id as request_id',
                'orr.distance_km',
                'orr.trip_amount',
                'orr.requested_at',
                'o.id as order_id',
                'o.order_no',
                'o.created_at as order_created_at',
                'o.delivered_at'
            )
            ->get()
            ->map(function ($row) use ($rider) {

                // Get materials for the order
                $materials = DB::table('order_items as oi')
                    ->join('products as p', 'p.id', '=', 'oi.product_id')
                    ->where('oi.order_id', $row->order_id)
                    ->select(
                        'p.category',
                        'p.subcategory',
                        'p.brand',
                        'oi.quantity',
                        'p.quantity_unit'
                    )
                    ->get();

                $distance = (float) $row->distance_km;
                $ratePerKm = (float) $rider->ratePerKm;
                $tripAmount = $distance * $ratePerKm;

                return [
                    'request_id' => $row->request_id,
                    'order_id'   => $row->order_id, 
                    'order_no'   => $row->order_no,
                    'materials'  => $materials,
                    'distance_km' => (float) $row->distance_km,
                    'trip_amount' => (float) $row->trip_amount,
                    'requested_date' => Carbon::parse($row->requested_at)->format('d/m/Y'),
                    'requested_time' => Carbon::parse($row->requested_at)->format('h:i A'),
                    'delivered_date' => Carbon::parse($row->delivered_at)->format('d/m/Y'),
                    'delivered_time' => Carbon::parse($row->delivered_at)->format('h:i A'),
                ];
            });

        return response()->json([
            'success' => true,
            // 'rider_work_status' => $workStatus,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }


    public function riderDeliveredOrderDetail($orderId)
    {
        $rider = auth('rider')->user();

        $riderData = DB::table('riders')
            ->where('id', $rider->id)
            ->first();

        // $workStatus = (bool) $riderData->work_status; 

        $row = DB::table('order_rider_requests as orr')
            ->join('orders as o', 'o.id', '=', 'orr.order_id')

            ->leftJoin('warehouses as w', function ($join) {
                $join->on('w.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'warehouse');
            })
            ->leftJoin('stores as s', function ($join) {
                $join->on('s.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'store');
            })

            ->leftJoin(
                'districts as pd',
                DB::raw('pd.id'),
                '=',
                DB::raw('COALESCE(w.district_id, s.district_id)')
            )
            ->leftJoin(
                'states as ps',
                DB::raw('ps.id'),
                '=',
                DB::raw('COALESCE(w.state_id, s.state_id)')
            )

            ->leftJoin('addresses as a', 'a.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'a.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'a.state_id')

            ->where('orr.rider_id', $rider->id)
            ->where('orr.status', 'completed')
            ->where('o.status', 'completed')
            ->where('o.id', $orderId)
            ->select(
                'orr.id as request_id',
                'orr.distance_km',
                'orr.trip_amount',
                'orr.requested_at',
                'o.id as order_id',
                'o.order_no',
                'o.created_at as order_created_at',
                'o.delivered_at',

                DB::raw("COALESCE(w.warehouse_name, s.store_name) as pickup_name"),
                DB::raw("COALESCE(w.door_no, s.door_no) as pickup_door_no"),
                DB::raw("COALESCE(w.street_name, s.street_name) as pickup_street"),
                DB::raw("COALESCE(w.area, s.area) as pickup_area"),
                DB::raw("COALESCE(w.city, s.city) as pickup_city"),
                DB::raw("COALESCE(w.pincode, s.pincode) as pickup_pincode"),
                DB::raw("COALESCE(w.mobile_number, s.mobile_number) as pickup_phone"),
                'pd.name as pickup_district',
                'ps.name as pickup_state',


                'a.name as drop_name',
                'a.mobile_number as drop_phone',
                'a.door_no as drop_door_no',
                'a.street_name as drop_street',
                'a.area as drop_area',
                'a.city as drop_city',
                'a.pincode as drop_pincode',
                'dd.name as drop_district',
                'ds.name as drop_state'
            )
            ->first();

        if (!$row) {
            return response()->json([
                'success' => false,
                'message' => 'Delivered order not found'
            ], 404);
        }

        // Get materials for the order
        $materials = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->where('oi.order_id', $row->order_id)
            ->select(
                'p.category',
                'p.subcategory',
                'p.brand',
                'oi.quantity',
                'p.quantity_unit'
            )
            ->get();

        $data = [
            'request_id' => $row->request_id,
            'order_id'   => $row->order_id, 
            'order_no'   => $row->order_no,
            'materials'  => $materials,
            'distance_km' => (float) $row->distance_km,
            'trip_amount' => (float) $row->trip_amount,
            'requested_date' => Carbon::parse($row->requested_at)->format('d/m/Y'),
            'requested_time' => Carbon::parse($row->requested_at)->format('h:i A'),
            'delivered_date' => Carbon::parse($row->delivered_at)->format('d/m/Y'),
            'delivered_time' => Carbon::parse($row->delivered_at)->format('h:i A'),

            'pickup_location' => [
                'name'     => $row->pickup_name,
                'phone'    => $row->pickup_phone,
                'door_no'  => $row->pickup_door_no,
                'street'   => $row->pickup_street,
                'area'     => $row->pickup_area,
                'city'     => $row->pickup_city,
                'district' => $row->pickup_district,
                'state'    => $row->pickup_state,
                'pincode'  => $row->pickup_pincode,
            ],

            'drop_location' => [
                'name'     => $row->drop_name,
                'phone'    => $row->drop_phone,
                'door_no'  => $row->drop_door_no,
                'street'   => $row->drop_street,
                'area'     => $row->drop_area,
                'city'     => $row->drop_city,
                'district' => $row->drop_district,
                'state'    => $row->drop_state,
                'pincode'  => $row->drop_pincode,
            ]
        ];

        return response()->json([
            'success' => true,
            // 'rider_work_status' => $workStatus,
            'data'    => $data
        ]);
    }

    public function riderEarnings(Request $request)
    {
        $rider = auth('rider')->user();

        $query = DB::table('order_rider_requests as orr')
            ->join('orders as o', 'o.id', '=', 'orr.order_id')
            ->where('orr.rider_id', $rider->id)
            ->where('orr.status', 'completed')
            ->where('o.status', 'completed');

        if (
            !$request->filled('date') &&
            !$request->filled('from_date') &&
            !$request->filled('to_date') &&
            !$request->filled('month') &&
            !$request->filled('year')
        ) {
            $query->whereDate('o.delivered_at', Carbon::today());
        }

        /**
         |--------------------------------------------------------------------------
        | SINGLE DATE FILTER
        |--------------------------------------------------------------------------
        | ?date=2026-01-24
        */
        if ($request->filled('date')) {
            $query->whereDate('o.delivered_at', $request->date);
        }

        /**
         |--------------------------------------------------------------------------
        | DATE RANGE FILTER
        |--------------------------------------------------------------------------
        | ?from_date=2026-01-01&to_date=2026-01-31
        */
        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween('o.delivered_at', [
                $request->from_date . ' 00:00:00',
                $request->to_date . ' 23:59:59'
            ]);
        }

        if ($request->filled('month') && $request->filled('year')) {
            $query->whereMonth('o.delivered_at', $request->month)
                ->whereYear('o.delivered_at', $request->year);
        }

        if ($request->filled('year') && !$request->filled('month')) {
            $query->whereYear('o.delivered_at', $request->year);
        }

        $stats = $query->select(
            DB::raw('COUNT(*) as total_trips'),
            DB::raw('SUM(orr.trip_amount) as total_earnings'),
            DB::raw('SUM(orr.distance_km) as total_distance')
        )->first();

        return response()->json([
            'success' => true,
            'filters' => $request->only(['date','from_date','to_date','month','year']),
            'data' => [
                'total_earnings' => (float) ($stats->total_earnings ?? 0),
                'total_trips'    => (int)   ($stats->total_trips ?? 0),
                'total_distance' => round((float) ($stats->total_distance ?? 0), 2),
            ]
        ]);
    }

    public function riderHome()
    {
        $rider = auth('rider')->user();

        if (!$rider) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $today = Carbon::today();

        $activeNotificationCount = DB::table('notifications')
            ->where('user_type', 'rider')
            ->where('user_id', $rider->id)
            ->where('status', 'active')
            ->count();

        $totalCompletedOrders = DB::table('order_rider_requests')
            ->where('rider_id', $rider->id)
            ->where('status', 'completed')
            ->count();

        $todayDeliveredOrders = DB::table('orders')
            ->where('rider_id', $rider->id)
            ->where('status', 'completed')
            ->whereDate('delivered_at', $today)
            ->count();

        $todayDistanceCovered = DB::table('order_rider_requests')
            ->where('rider_id', $rider->id)
            ->where('status', 'completed')
            ->whereDate('updated_at', $today)
            ->sum('distance_km');

        $lastWeekStart = Carbon::now()->subWeek()->startOfWeek(Carbon::MONDAY);
        $lastWeekEnd   = Carbon::now()->subWeek()->endOfWeek(Carbon::SUNDAY);
        $lastWeekRevenue = DB::table('order_rider_requests')
            ->where('rider_id', $rider->id)
            ->where('status', 'completed')
            ->whereBetween('updated_at', [$lastWeekStart, $lastWeekEnd])
            ->sum('trip_amount');

        return response()->json([
            'success' => true,
            'data' => [
                'rider' => [
                    'name' => $rider->name,
                    'vehicle_number' => $rider->vehicle_number,
                    'is_available' => (bool) $rider->is_available,
                ],
                'notifications' => [
                    'active_count' => $activeNotificationCount
                ],
                'stats' => [
                    'total_completed_orders' => $totalCompletedOrders,
                    'today_delivered_orders' => $todayDeliveredOrders,
                    'today_distance_km' => round($todayDistanceCovered, 2),
                    'last_week_revenue' => round($lastWeekRevenue, 2),
                ]
            ]
        ]);
    }

}
