<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\Product;
use App\Models\QuantityUnit;
use App\Models\Store;
use App\Models\StoreProduct;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Tymon\JWTAuth\Facades\JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

class StoreController extends Controller
{
    // ================= REGISTER =================
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'owner_name' => 'required|string|max:255',
            'email' => 'required|email|unique:stores,email',
            'mobile_number' => 'required|unique:stores,mobile_number',
            'password' => 'required|min:8',

            'store_name' => 'required|string|max:255',
            'door_no' => 'required',
            'street_name' => 'required',
            'area' => 'required',
            'city' => 'required',
            'state_id' => 'required|exists:states,id',
            'district_id' => 'required|exists:districts,id',
            'pincode' => 'required',
            'country' => 'required',

            'pan_number' => 'required|unique:stores,pan_number',
            'aadhar_number' => 'required|digits:12|unique:stores,aadhar_number',

            'business_type' => 'nullable|in:Individual,Proprietorship,Partnership,Private Limited',
            'business_status' => 'required|in:registered,unregistered',

            'gst_number' => 'nullable|string|max:50|required_if:business_status,registered|unique:stores,gst_number',
            'trade_license_number' => 'nullable|string|max:255|required_if:business_status,registered',

            'products' => 'required|array|min:1',
            'products.*.category_id' => 'required|exists:categories,id',
            'products.*.subcategory_id' => 'nullable|exists:subcategories,id',
            'products.*.quantity_unit_id' => 'required|exists:quantity_units,id',
            'products.*.brand_id' => 'nullable|exists:brands,id',
            'products.*.price' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $otp = '1111';
        $registrationToken = Str::uuid()->toString();

        Cache::put(
            'store_register_' . $registrationToken,
            [
                'otp' => $otp,
                'otp_created_at' => now(),
                'data' => $request->all()
            ],
            now()->addMinutes(10)
        );

        return response()->json([
            'success' => true,
            'message' => 'OTP generated. Please verify.',
            'registration_token' => $registrationToken,
            'otp' => $otp // TEMP
        ]);
    }


    public function verifyOtp(Request $request)
    {
        $request->validate([
            'registration_token' => 'required',
            'otp' => 'required'
        ]);

        $cacheKey = 'store_register_' . $request->registration_token;
        $cached = Cache::get($cacheKey);

        if (!$cached) {
            return response()->json([
                'success' => false,
                'message' => 'Session expired. Please register again.'
            ], 400);
        }

        if (
            $cached['otp'] !== $request->otp ||
            now()->diffInMinutes($cached['otp_created_at']) > 5
        ) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired OTP'
            ], 400);
        }

        $data = $cached['data'];

        $store = Store::create([
            'owner_name' => $data['owner_name'],
            'email' => $data['email'],
            'mobile_number' => $data['mobile_number'],
            'password' => Hash::make($data['password']),

            'store_name' => $data['store_name'],
            'door_no' => $data['door_no'],
            'street_name' => $data['street_name'],
            'area' => $data['area'],
            'city' => $data['city'],
            'state_id' => $data['state_id'],
            'district_id' => $data['district_id'],
            'pincode' => $data['pincode'],
            'country' => $data['country'],

            'pan_number' => $data['pan_number'],
            'aadhar_number' => $data['aadhar_number'],

            'status' => 'pending_admin',
            'business_type' => $data['business_type'] ?? null,
            'business_status' => $data['business_status'],
            'gst_number' => $data['business_status'] === 'registered' ? $data['gst_number'] : null,
            'trade_license_number' => $data['business_status'] === 'registered' ? $data['trade_license_number'] : null,

            'admin_remark' => null,
            'approved_at' => null,
            'approved_by' => null,
        ]);

        if (!empty($data['products'])) {
            foreach ($data['products'] as $p) {

                $validUnit = QuantityUnit::where('id', $p['quantity_unit_id'])
                    ->whereJsonContains('category_ids', (string) $p['category_id'])
                    ->exists();

                if (!$validUnit) {
                    continue;
                }

                $productQuery = Product::where('status', 'active')
                    ->where('category_id', $p['category_id'])
                    ->where('quantity_unit_id', $p['quantity_unit_id']);

                if (!empty($p['subcategory_id'])) {
                    $productQuery->where('subcategory_id', $p['subcategory_id']);
                } else {
                    $productQuery->whereNull('subcategory_id');
                }

                if (!empty($p['brand_id'])) {
                    $productQuery->where('brand_id', $p['brand_id']);
                } else {
                    $productQuery->whereNull('brand_id');
                }

                $product = $productQuery->first();

                StoreProduct::create([
                    'store_id'         => $store->id,
                    'product_id'       => $product?->id,
                    'category_id'      => $p['category_id'],
                    'subcategory_id'   => $p['subcategory_id'] ?? null,
                    'quantity_unit_id' => $p['quantity_unit_id'],
                    'brand_id'         => $p['brand_id'] ?? null,
                    'price'            => $p['price'],
                    'status'           => 'pending',
                ]);
            }
        }

        Address::create([
            'user_type'     => 'store',
            'user_id'       => $store->id,
            'name'          => $data['owner_name'],
            'mobile_number' => $data['mobile_number'],
            'door_no'       => $data['door_no'],
            'street_name'   => $data['street_name'],
            'area'          => $data['area'],
            'city'          => $data['city'],
            'district_id'   => $data['district_id'],
            'state_id'      => $data['state_id'],
            'pincode'       => $data['pincode'],
            'is_default'    => 1
        ]);

        Cache::forget($cacheKey);

        return response()->json([
            'success' => true,
            'message' => 'OTP verified. Store registration completed.',
            'store_id' => $store->id
        ]);
    }


    // ================= LOGIN =================
    public function login(Request $request)
    {
        $request->validate([
            'login' => 'required',
            'password' => 'required|string|min:4'
        ]);

        $store = Store::where('email', $request->login)
            ->orWhere('mobile_number', $request->login)
            ->first();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found'
            ], 404);
        }

        if (!Hash::check($request->password, $store->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid password'
            ], 401);
        }

        if ($store->status !== 'active') {

            $message = match ($store->status) {
                'pending_otp'   => 'OTP verification required',
                'pending_admin' => 'Account pending admin approval',
                'rejected'      => 'Account rejected by admin',
                default         => 'Account inactive',
            };

            return response()->json([
                'success' => false,
                'message' => $message
            ], 403);
        }

        if ($request->filled('fcm_token')) {
            DB::table('fcm_tokens')->updateOrInsert(
                [
                    'user_type' => 'store',
                    'user_id'   => $store->id,
                ],
                [
                    'fcm_token'     => $request->fcm_token,
                    'notify_status' => 'active',
                    'device_type'   => $request->device_type ?? 'android',
                    'updated_at'    => now(),
                    'created_at'    => now()
                ]
            );
        }

        $token = JWTAuth::customClaims([
            'exp' => now()->addDays(30)->timestamp
        ])->fromUser($store);

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'token' => $token,
            'token_type' => 'bearer',
            'expires_in' => JWTAuth::factory()->getTTL() * 60,  
            'store' => [
                'id' => $store->id,
                'name' => $store->name,
                'email' => $store->email,
                'mobile_number' => $store->mobile_number,
                'city' => $store->city
            ]
        ]);
    }

    // ================= PROFILE =================
    public function profile()
    {
        return response()->json([
            'success' => true,
            'store' => auth('store')->user()
        ]);
    }

    public function changePassword(Request $request)
    {
        $store = auth('store')->user();

        $request->validate([
            'current_password'      => 'required|string',
            'new_password'          => 'required|string|min:8|confirmed',
            // 'confirmed' expects a field named new_password_confirmation
        ]);

        if (!Hash::check($request->current_password, $store->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect.'
            ], 400);
        }

        $store->password = Hash::make($request->new_password);
        $store->save();

        JWTAuth::invalidate(JWTAuth::getToken());

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully. You have been logged out.'
        ]);
    }

    public function accountDetails()
    {
        $store = auth('store')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'owner_name'     => $store->owner_name,
                'store_name' => $store->store_name,
                'mobile_number'  => $store->mobile_number,

                'logo_url' => $store->logo
                    ? url($store->logo)
                    : null,

                'logo_initial' => strtoupper(
                    substr($store->store ?? $store->owner_name, 0, 1)
                ),
            ]
        ]);
    }

    public function updateLogo(Request $request)
    {
        $store = auth('store')->user();

        $request->validate([
            'logo' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);

        if ($store->logo && file_exists(public_path($store->logo))) {
            unlink(public_path($store->logo));
        }

        $file = $request->file('logo');
        $filename = 'store_' . $store->id . '_' . time() . '.' . $file->getClientOriginalExtension();

        $uploadPath = 'uploads/store/logos';
        $file->move(public_path($uploadPath), $filename);

        $store->update([
            'logo' => $uploadPath . '/' . $filename
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Logo updated successfully',
            'logo_url' => url($uploadPath . '/' . $filename)
        ]);
    }

    public function ownerDetails(Request $request)
    {
        $store = auth('store')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'owner_name'    => $store->owner_name,
                'mobile_number' => $store->mobile_number,
                'email'         => $store->email,
            ]
        ]);
    }

    public function businessDetails()
    {
        $store = auth('store')->user()->load(['state', 'district']);

        return response()->json([
            'success' => true,
            'data' => [
                'store_name'  => $store->store_name,
                'door_no'         => $store->door_no,
                'street_name'     => $store->street_name,
                'area'            => $store->area,
                'city'            => $store->city,
                'district_name'   => $store->district->name ?? null,
                'state_name'      => $store->state->name ?? null,
                'pincode'         => $store->pincode,
                'business_type'   => $store->business_type,
                'business_status' => $store->business_status,
            ]
        ]);
    }

    public function legalDocuments()
    {
        $store = auth('store')->user();
        return response()->json([
            'success' => true,
            'data' => [
                'gst_number'           => $store->gst_number,
                'trade_license_number' => $store->trade_license_number,
                'pan_number'           => $store->pan_number,
            ]
        ]);
    }
    
    public function bankDetails()
    {
        $store = auth('store')->user()->load('bank');

        return response()->json([
            'success' => true,
            'bank_status' => $store->bank_status,
            'has_pending_update' => !empty($store->pending_bank_data),

            'data' => [
                'bank_holder_name'    => $store->bank_holder_name,
                'bank_id'             => $store->bank_id,
                'bank_name'           => $store->bank->name ?? null,
                'bank_account_number' => $store->bank_account_number,
                'ifsc_code'           => $store->ifsc_code,
                'micr_no'             => $store->micr_no,
                'upi_id'              => $store->upi_id,
                'passbook_file'       => $store->passbook_file 
                    ? asset('storage/' . $store->passbook_file) 
                    : null,
            ],

            'pending_data' => $store->pending_bank_data,
            'admin_remark' => $store->admin_remark,
        ]);
    }

    public function saveBankDetails(Request $request)
    {
        $store = auth('store')->user();

        if ($store->bank_id) {
            return response()->json([
                'success' => false,
                'message' => 'Bank details already exist. Please update instead.'
            ], 403);
        }

        $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'bank_holder_name' => 'required|string|max:255',
            'bank_account_number' => 'required|confirmed',
            'bank_account_number_confirmation' => 'required',
            'ifsc_code' => 'required|string|max:255',
            'micr_no' => 'nullable|string|max:50',
            'upi_id' => 'nullable|string|max:100',
            'passbook_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024',
        ]);

        $file = $request->file('passbook_file');
        $filename = 'passbook_' . $store->id . '_' . time() . '.' . $file->getClientOriginalExtension();
        $uploadPath = 'store/passbooks';
        $file->move(public_path($uploadPath), $filename);

        $store->update([
            'pending_bank_data' => [
                'bank_id' => $request->bank_id,
                'bank_holder_name' => $request->bank_holder_name,
                'bank_account_number' => $request->bank_account_number,
                'ifsc_code' => $request->ifsc_code,
                'micr_no' => $request->micr_no,
                'upi_id' => $request->upi_id,
                'passbook_file' => $uploadPath . '/' . $filename,
            ],
            'bank_status' => 'pending',
            'admin_remark' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Bank details submitted for admin approval'
        ]);
    }

    public function updateBankDetails(Request $request)
    {
        $store = auth('store')->user();

        if ($store->bank_status === 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Already waiting for admin approval'
            ], 403);
        }

        $request->validate([
            'bank_id' => 'sometimes|exists:banks,id',
            'bank_holder_name' => 'sometimes|string|max:255',
            'bank_account_number' => 'sometimes|confirmed',
            'bank_account_number_confirmation' => 'required_with:bank_account_number',
            'ifsc_code' => 'sometimes|string|max:255',
            'micr_no' => 'nullable|string|max:50',
            'upi_id' => 'nullable|string|max:100',
            'passbook_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024',
        ]);

        $pending = $store->pending_bank_data ?? [];

        if ($request->hasFile('passbook_file')) {
            // delete old pending passbook if exists
            if (!empty($pending['passbook_file']) && file_exists(public_path($pending['passbook_file']))) {
                unlink(public_path($pending['passbook_file']));
            }

            $file = $request->file('passbook_file');
            $filename = 'passbook_' . $store->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'store/passbooks';
            $file->move(public_path($uploadPath), $filename);

            $pending['passbook_file'] = $uploadPath . '/' . $filename;
        }

        $store->update([
            'pending_bank_data' => [
                'bank_id' => $request->bank_id ?? $store->bank_id,
                'bank_holder_name' => $request->bank_holder_name ?? $store->bank_holder_name,
                'bank_account_number' => $request->bank_account_number ?? $store->bank_account_number,
                'ifsc_code' => $request->ifsc_code ?? $store->ifsc_code,
                'micr_no' => $request->micr_no ?? $store->micr_no,
                'upi_id' => $request->upi_id ?? $store->upi_id,
                'passbook_file' => $pending['passbook_file'] ?? $store->passbook_file,
            ],
            'bank_status' => 'pending',
            'admin_remark' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Updated bank details sent for admin approval'
        ]);
    }


    public function quotationDetails()
    {
        $store = auth('store')->user();

        $products = StoreProduct::with([
                'category:id,name',
                'subcategory:id,name',
                'brand:id,name',
                'quantityUnit:id,name'
            ])
            ->where('store_id', $store->id)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $products->map(function ($item) {
                return [
                    'category_name'   => $item->category->name ?? null,
                    'sub_category'    => $item->subcategory->name ?? null,
                    'brand_name'      => $item->brand->name ?? null,
                    'unit'            => $item->quantityUnit->name ?? null,
                    'price'           => $item->price,
                    'status'          => $item->status
                ];
            })
        ]);
    }

    public function storeCategoryPage()
    {
        $store = auth('store')->user();

        $categories = \DB::table('store_products')
            ->join('categories', 'categories.id', '=', 'store_products.category_id')
            ->where('store_products.store_id', $store->id)
            ->groupBy(                
                'store_products.category_id',
                'categories.name',
                'categories.image'
            )
            ->select(
                'store_products.category_id',
                'categories.name as category_name',
                'categories.image as category_image',
                \DB::raw("
                    SUM(CASE 
                        WHEN store_products.status = 'approved' 
                        THEN 1 ELSE 0 
                    END) as approved_count
                "),
                \DB::raw("
                    MAX(store_products.category_active) as category_active
                ")
            )
            ->orderBy('categories.name')
            ->get()
            ->map(function ($item) {
                return [
                    'category_id' => $item->category_id,
                    'name'   => $item->category_name,
                    'image'  => $item->category_image
                        ? asset($item->category_image)
                        : asset('images/no-image.png'),
                    'ui_status'   => $item->approved_count > 0 ? 'enabled' : 'disabled',

                    'category_status' => $item->category_active == 1 ? 'active' : 'inactive',
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $categories
        ], 200);
    }

    // public function categoryProducts(Request $request)
    // {
    //     $store = auth('store')->user();

    //     $request->validate([
    //         'category' => 'required|string'
    //     ]);

    //     $products = \DB::table('store_products')
    //         ->leftJoin('categories', 'categories.id', '=', 'store_products.category_id')
    //         ->leftJoin('subcategories', 'subcategories.id', '=', 'store_products.subcategory_id')
    //         ->leftJoin('brands', 'brands.id', '=', 'store_products.brand_id')
    //         ->where('store_products.store_id', $store->id)
    //         ->where('categories.name', $request->category)
    //         ->select(
    //             'brands.name as brand_name',
    //             'categories.name as category_name',
    //             'subcategories.name as subcategory_name',
    //             'store_products.quantity_unit',
    //             'store_products.price',
    //             'store_products.status'
    //         )
    //         ->orderBy('brands.name')
    //         ->get();

    //     return response()->json([
    //         'success'  => true,
    //         'category' => $request->category,
    //         'data'     => $products
    //     ]);
    // }

    public function getStoreProductForEdit($id)
    {
        $store = auth('store')->user();

        $product = \DB::table('store_products as sp')
            ->leftJoin('categories as c', 'c.id', '=', 'sp.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'sp.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'sp.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'sp.quantity_unit_id')
            ->where('sp.id', $id)
            ->where('sp.store_id', $store->id)
            ->select(
                'sp.id',
                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit',
                'sp.price',
                'sp.status'
            )
            ->first();

        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        $statusMessage = $product->status === 'pending'
            ? 'Waiting for admin approval'
            : $product->status;

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $product->id,
                'category' => $product->category,
                'subcategory' => $product->subcategory,
                'brand' => $product->brand,
                'quantity_unit' => $product->quantity_unit,
                'price' => $product->price,
                'status' => $statusMessage

            ]
        ]);
    }
    public function categoryProductsById($category_id)
    {
        $store = auth('store')->user();

        $products = \DB::table('store_products as sp')
            ->leftJoin('products as p', function ($join) {
                $join->on('p.category_id', '=', 'sp.category_id')
                    ->on('p.quantity_unit_id', '=', 'sp.quantity_unit_id')
                    ->whereColumn('p.subcategory_id', 'sp.subcategory_id')
                    ->whereColumn('p.brand_id', 'sp.brand_id');
            })
            ->leftJoin('categories as c', 'c.id', '=', 'sp.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'sp.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'sp.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'sp.quantity_unit_id')
            ->where('sp.store_id', $store->id)
            ->where('sp.category_id', $category_id)
            ->select(
                'sp.id as store_product_id',
                'c.name as category_name',
                's.name as subcategory_name',
                'b.name as brand_name',
                'q.name as quantity_unit',
                'sp.price',
                'sp.status',
                'sp.product_active',
                'sp.admin_remark',
                'p.image as product_image'
            )
            ->orderBy('sp.created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $products->map(function ($item) {
                return [
                    'id' => $item->store_product_id,
                    'category' => $item->category_name,
                    'subcategory' => $item->subcategory_name,
                    'brand' => $item->brand_name,
                    'unit' => $item->quantity_unit,
                    'price' => $item->price,
                    'status' => $item->status,
                    'admin_remark' => $item->admin_remark,
                    'active' => (bool) $item->product_active,

                    'image' => $item->product_image
                        ? asset($item->product_image)
                        : null
                ];
            })
        ]);
    }


    public function updateStoreProduct(Request $request, $id)
    {
        $store = auth('store')->user();

        $request->validate([
            'price' => 'required|numeric|min:0',
        ]);

        $storeProduct = \DB::table('store_products')
            ->where('id', $id)
            ->where('store_id', $store->id)
            ->first();

        if (!$storeProduct) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        \DB::table('store_products')
            ->where('id', $id)
            ->update([
                'price' => $request->price,
                'status' => 'pending',
                'admin_remark' => null,
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Price updated and sent for admin approval'
        ]);
    }


    public function addProducts(Request $request)
    {
        $store = auth('store')->user();

        $request->validate([
            'products' => 'required|array|min:1',
            'products.*.category_id' => 'required|exists:categories,id',
            'products.*.subcategory_id' => 'nullable|exists:subcategories,id',
            'products.*.quantity_unit_id' => 'required|exists:quantity_units,id',
            'products.*.brand_id' => 'nullable|exists:brands,id',
            'products.*.price' => 'required|numeric|min:0',
        ]);

        foreach ($request->products as $p) {

            $validUnit = QuantityUnit::where('id', $p['quantity_unit_id'])
                ->whereJsonContains('category_ids', (string) $p['category_id'])
                ->exists();

            if (!$validUnit) {
                continue;
            }
            $productQuery = Product::where('status', 'active')
                ->where('category_id', $p['category_id'])
                ->where('quantity_unit_id', $p['quantity_unit_id']);

            if (!empty($p['subcategory_id'])) {
                $productQuery->where('subcategory_id', $p['subcategory_id']);
            } else {
                $productQuery->whereNull('subcategory_id');
            }

            if (!empty($p['brand_id'])) {
                $productQuery->where('brand_id', $p['brand_id']);
            } else {
                $productQuery->whereNull('brand_id');
            }

            $product = $productQuery->first();

            StoreProduct::create([
                'store_id'     => $store->id,
                'product_id'       => $product?->id,
                'category_id'      => $p['category_id'],
                'subcategory_id'   => $p['subcategory_id'] ?? null,
                'quantity_unit_id' => $p['quantity_unit_id'],
                'brand_id'         => $p['brand_id'] ?? null,
                'price'            => $p['price'],
                'status'           => 'pending',
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Product(s) added and sent for admin approval'
        ]);
    }

    public function toggleStoreCategoryActive($category_id)
    {
        $store = auth('store')->user();

        // Get current state
        $current = \DB::table('store_products')
            ->where('store_id', $store->id)
            ->where('category_id', $category_id)
            ->value('category_active');

        if ($current === null) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        }

        $newStatus = !$current;

        \DB::table('store_products')
            ->where('store_id', $store->id)
            ->where('category_id', $category_id)
            ->update([
                'category_active' => $newStatus
            ]);

        return response()->json([
            'success' => true,
            'active' => $newStatus
        ]);
    }

    public function deleteStoreCategory($category_id)
    {
        $store = auth('store')->user();

        \DB::table('store_products')
            ->where('store_id', $store->id)
            ->where('category_id', $category_id)
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'Category removed'
        ]);
    }

    public function toggleStoreProductActive($store_product_id)
    {
        $store = auth('store')->user();

        // Get current state
        $current = \DB::table('store_products')
            ->where('store_id', $store->id)
            ->where('id', $store_product_id)
            ->value('product_active');

        if ($current === null) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        $newStatus = !$current;

        \DB::table('store_products')
            ->where('store_id', $store->id)
            ->where('id', $store_product_id)
            ->update([
                'product_active' => $newStatus
            ]);

        return response()->json([
            'success' => true,
            'active'  => $newStatus
        ]);
    }



    public function deleteStoreProduct($store_product_id)
    {
        $store = auth('store')->user();

        $deleted = \DB::table('store_products')
            ->where('store_id', $store->id)
            ->where('id', $store_product_id)
            ->delete();

        if (!$deleted) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Product removed'
        ]);
    }


    
    // ================= LOGOUT =================
    public function logout()
    {
        try {
            JWTAuth::invalidate(JWTAuth::getToken());

            return response()->json([
                'success' => true,
                'message' => 'Logged out successfully'
            ]);
        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Logout failed'
            ], 500);
        }
    }

    // ================= REFRESH =================
    public function refresh()
    {
        try {
            $newToken = JWTAuth::refresh(JWTAuth::getToken());

            return response()->json([
                'success' => true,
                'token' => $newToken
            ]);
        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Token refresh failed'
            ], 401);
        }
    }

    // ================= FORGOT PASSWORD - SEND OTP =================
    public function forgotPassword(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10'
        ]);

        $store = Store::where('mobile_number', $request->mobile_number)->first();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Mobile number not registered'
            ], 404);
        }

        $otp = '1111'; // TEST OTP

        $store->update([
            'otp' => $otp,
            'otp_created_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'OTP sent successfully (TEST OTP: 1111)'
        ]);
    }

    // ================= FORGOT PASSWORD - VERIFY OTP =================
    public function verifyForgotOtp(Request $request)
    {
        $request->validate([
            'otp' => 'required'
        ]);

        $store = Store::where('otp', $request->otp)
            ->whereNotNull('otp_created_at')
            ->orderBy('otp_created_at', 'desc')
            ->first();

        if (
            !$store ||
            now()->diffInMinutes($store->otp_created_at) > 5
        ) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired OTP'
            ], 400);
        }

        return response()->json([
            'success' => true,
            'message' => 'OTP verified successfully',
            'store_id' => $store->id
        ]);
    }

    // ================= RESET PASSWORD =================
    public function resetPassword(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10|exists:stores,mobile_number',
            'password' => 'required|min:8|confirmed',
        ]);

        $store = Store::where('mobile_number', $request->mobile_number)->first();

        $store->update([
            'password' => Hash::make($request->password),
            'otp' => null,
            'otp_created_at' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully'
        ]);
    }

    public function notifications()
    {
        if (!auth('store')->check()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $storeId = auth('store')->id();

        $notifications = DB::table('notifications')
            ->where('user_type', 'store')
            ->where('user_id', $storeId)
            ->whereDate('created_at', '>=', now()->subDays(15))
            ->orderBy('created_at', 'desc')
            ->get();

        $grouped = [];

        foreach ($notifications as $n) {

            $createdAt = Carbon::parse($n->created_at);
            $link = $n->link ? json_decode($n->link, true) : null;

            if ($createdAt->isToday()) {
                $groupKey = 'Today';
            } elseif ($createdAt->isYesterday()) {
                $groupKey = 'Yesterday';
            } else {
                $groupKey = $createdAt->format('Y-m-d');
            }

            $grouped[$groupKey][] = [
                'id'       => $n->id,
                'title'    => $n->title,
                'content'  => $n->content,
                'route'    => $link['route'] ?? null,
                'route_id' => $link['route_id'] ?? null,
                'status'   => $n->status,
                'time'     => $createdAt->format('h:i A'),
            ];
        }

        return response()->json([
            'success' => true,
            'days'    => count($grouped),
            'data'    => $grouped
        ]);
    }

    public function activeNotificationsCount(Request $request)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $activeCount = DB::table('notifications')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('status', 'active')
            ->count();

        return response()->json([
            'success' => true,
            'active_count' => $activeCount
        ]);
    }

    public function toggleNotificationStatus(Request $request, $notificationId)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $notification = DB::table('notifications')
            ->where('id', $notificationId)
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->first();

        if ($notification->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Notification is already inactive'
            ], 400);
        }

        DB::table('notifications')
            ->where('id', $notificationId)
            ->update([
                'status'     => 'inactive',
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Notification deactivated successfully',
            'data' => [
                'notification_id' => $notificationId,
                'status' => 'inactive',
            ]
        ], 200);

    }
}
