<?php

namespace App\Http\Controllers;
use App\Mail\OrderTakenMail;
use App\Models\Store;
use App\Models\Warehouse;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Mail;


class StoreOrderController extends Controller
{
    public function getNearbyCustomerOrders(Request $request)
    {
        $store = auth('store')->user();
        $radiusKm = $request->input('radius', 10);

        if (!$store->latitude || !$store->longitude) {
            return response()->json([
                'success' => false,
                'message' => 'Store location not available'
            ], 400);
        }

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.rider_id',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name',
                'a.latitude',
                'a.longitude',
                'o.rejected_by',

                DB::raw("
                    IF(
                        o.with_shipping = 1,
                        (
                            6371 * acos(
                                cos(radians(?)) *
                                cos(radians(a.latitude)) *
                                cos(radians(a.longitude) - radians(?)) +
                                sin(radians(?)) *
                                sin(radians(a.latitude))
                            )
                        ),
                        NULL
                    ) AS distance
                ")
            )
            ->addBinding($store->latitude)
            ->addBinding($store->longitude)
            ->addBinding($store->latitude)
            ->where('o.user_type', 'customer')
            ->where('o.status', 'pending_assignment')

            ->whereExists(function ($q) use ($store) {
                $q->select(DB::raw(1))
                    ->from('order_items as oi')
                    ->join('store_products as sp', 'sp.product_id', '=', 'oi.product_id')
                    ->whereRaw('oi.order_id = o.id')
                    ->where('sp.store_id', $store->id)
                    ->where('sp.product_active', 1)
                    ->where('sp.status', 'approved');
            })

            ->where(function ($query) use ($store) {
                $currentUserJson = json_encode([
                    'user_id' => $store->id,
                    'user_type' => 'store'
                ]);

                $query->whereNull('o.rejected_by')
                    ->orWhereRaw("JSON_CONTAINS(o.rejected_by, ?, '$') = 0", [$currentUserJson]);
            })


            ->havingRaw("
                (o.with_shipping = 1 AND distance <= ?)
                OR
                (o.with_shipping = 0)
            ", [$radiusKm])
            ->orderByRaw("IF(o.payment_method = 'COD', o.created_at, o.paid_at) DESC")

            ->get();

        $orders = $orders->map(function ($order) {

            $dateTime = $order->payment_method === 'COD'
                ? $order->created_at
                : $order->paid_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'oi.id',
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category' => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand' => $item->brand,
                        'quantity' => $item->quantity,
                        'unit' => $item->quantity_unit,
                        'image' => $item->image
                            ? asset($item->image)
                            : null
                    ];
                });


            return [
                'order_id'   => $order->id,
                'order_no'   => $order->order_no,
                'status'     => $order->status,
                'shipping_type'  => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'rider_id'   => $order->rider_id,
                'rider_code' => $order->rider_code,

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'      => $order->with_shipping ? $order->door_no : null,
                'street_name'  => $order->with_shipping ? $order->street_name : null,
                'area'         => $order->with_shipping ? $order->area : null,
                'city'         => $order->with_shipping ? $order->city : null,
                'district'     => $order->with_shipping ? $order->district_name : null,
                'state'        => $order->with_shipping ? $order->state_name : null,
                'pincode'      => $order->with_shipping ? $order->pincode : null,

                'distance_km' => $order->distance,

                'order_items' => $items
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    // public function approveOrder($order_id)
    // {
    //     $warehouse = auth('warehouse')->user();

    //     $order = DB::table('orders')->where('id', $order_id)->first();

    //     if (!$order) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Order not found'
    //         ], 404);
    //     }

    //     if ($order->with_shipping == 0) {

    //         if (!$order->rider_id) {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Rider not assigned for this order'
    //             ], 400);
    //         }

    //         DB::beginTransaction();

    //         try {
    //             DB::table('orders')->where('id', $order_id)->update([
    //                 'status' => 'rider_accepted',
    //                 'approved_by' => $warehouse->id,
    //                 'approved_by_user_type' => 'warehouse',
    //                 'approved_at' => now(),
    //                 'updated_at' => now(),
    //             ]);

    //             $riderRequest = DB::table('order_rider_requests')
    //                 ->where('order_id', $order_id)
    //                 ->where('rider_id', $order->rider_id)
    //                 ->first();

    //             if ($riderRequest) {
    //                 DB::table('order_rider_requests')
    //                     ->where('id', $riderRequest->id)
    //                     ->update([
    //                         'status' => 'accepted',
    //                         'responded_at' => now(),
    //                         'updated_at' => now(),
    //                     ]);
    //             } else {
    //                 DB::table('order_rider_requests')->insert([
    //                     'order_id'     => $order_id,
    //                     'rider_id'     => $order->rider_id,
    //                     'warehouse_id' => $warehouse->id,
    //                     'status'       => 'accepted',
    //                     'requested_at' => now(),
    //                     'responded_at' => now(),
    //                     'created_at'   => now(),
    //                     'updated_at'   => now(),
    //                 ]);
    //             }

    //             DB::commit();

    //             return response()->json([
    //                 'success' => true,
    //                 'message' => 'Order approved and rider accepted successfully'
    //             ]);

    //         } catch (\Exception $e) {
    //             DB::rollBack();

    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Something went wrong',
    //                 'error' => $e->getMessage()
    //             ], 500);
    //         }
    //     }

    //     DB::table('orders')->where('id', $order_id)->update([
    //         'status' => 'order_taken',
    //         'approved_by' => $warehouse->id,
    //         'approved_by_user_type' => 'warehouse',
    //         'approved_at' => now(),
    //         'updated_at' => now(),
    //     ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Order approved successfully'
    //     ]);
    // }

    private function calculateDistanceKm($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371;

        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);

        $a = sin($dLat / 2) * sin($dLat / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLon / 2) * sin($dLon / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

        return round($earthRadius * $c, 2);
    }

    private function sendOrderTakenMail($order)
    {
        if (!$order || !$order->approved_by || !$order->approved_by_user_type) {
            return;
        }

        $receiver = null;

        if ($order->approved_by_user_type === 'warehouse') {
            $receiver = Warehouse::find($order->approved_by);
        } elseif ($order->approved_by_user_type === 'store') {
            $receiver = Store::find($order->approved_by);
        }

        if (!$receiver || empty($receiver->email)) {
            return;
        }

        Mail::to($receiver->email)->send(
            new OrderTakenMail($order, $receiver, $order->approved_by_user_type)
        );
    }
    public function approveOrder($order_id)
    {
        $store = auth('store')->user();

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        DB::beginTransaction();

        try {
            if ($order->with_shipping == 0) {

                if (!$order->rider_id) {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Rider not assigned'
                    ], 400);
                }

                $rider = DB::table('riders')->where('id', $order->rider_id)->first();
                if (!$rider || !$rider->latitude || !$rider->longitude) {
                    throw new \Exception('Rider location not available');
                }

                if (!$store->latitude || !$store->longitude) {
                    throw new \Exception('store location not available');
                }

                $distanceKm = $this->calculateDistanceKm(
                    $rider->latitude,
                    $rider->longitude,
                    $store->latitude,
                    $store->longitude
                );

                $otp = rand(1000, 9999);

                DB::table('orders')->where('id', $order_id)->update([
                    'status'                => 'rider_accepted',
                    'approved_by'           => $store->id,
                    'approved_by_user_type' => 'store',
                    'approved_at'           => now(),
                    'updated_at'            => now(),
                ]);

                
                DB::table('order_rider_requests')->updateOrInsert(
                    [
                        'order_id' => $order_id,
                        'rider_id' => $order->rider_id
                    ],
                    [
                        'store_id'     => $store->id,
                        'latitude'     => $rider->latitude,
                        'longitude'    => $rider->longitude,
                        'distance_km'  => $distanceKm,
                        'status'       => 'active',
                        'otp'          => $otp,
                        'accepted_at'  => now(),
                        'requested_at' => now(),
                        'responded_at' => now(),
                        'updated_at'   => now(),
                    ]
                );

                DB::table('riders')
                    ->where('id', $order->rider_id)
                    ->update([
                        'work_status' => 1,
                        'updated_at'  => now(),
                    ]);
                

                DB::commit();

                $orderFresh = DB::table('orders')->where('id', $order_id)->first();
                $this->sendOrderTakenMail($orderFresh);

                return response()->json([
                    'success'     => true,
                    'message'     => 'Order approved (Without Shipping)',
                    'order_id'    => $order_id,
                    'otp'         => $otp,
                    'distance_km' => $distanceKm
                ]);
            }

            DB::table('orders')->where('id', $order_id)->update([
                'status'                => 'order_taken',
                'approved_by'           => $store->id,
                'approved_by_user_type' => 'store',
                'approved_at'           => now(),
                'updated_at'            => now(),
            ]);

            $orderItem = DB::table('order_items')
                ->where('order_id', $order_id)
                ->first();

            if (!$orderItem) {
                throw new \Exception('Order item not found');
            }

            $storeProduct = DB::table('store_products')
                ->where('product_id', $orderItem->product_id)
                ->where('store_id',  $store->id)
                ->first();

            if (!storeProduct) {
                throw new \Exception(
                    "store product not found for product_id={$orderItem->product_id}, store_id={ $store->id}"
                );
            }

            DB::table('order_items')
                ->where('id', $orderItem->id)
                ->update([
                    'approved_unit_price' => $storeProduct->price,
                    'total_price'         => $storeProduct->price * $orderItem->quantity,
                    'updated_at'          => now()
                ]);

            DB::commit();

            $orderFresh = DB::table('orders')->where('id', $order_id)->first();
            $this->sendOrderTakenMail($orderFresh);


            return response()->json([
                'success'  => true,
                'message'  => 'Order approved (With Shipping)',
                'order_id' => $order_id
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function rejectOrder(Request $request, $order_id)
    {
        $store = auth('store')->user();

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        $newEntry = [
            'user_id' => $store->id,
            'user_type' => 'store',
            'rejected_at' => Carbon::now()->toDateTimeString()
        ];

        $existing = $order->rejected_by ? json_decode($order->rejected_by, true) : [];

        $existing[] = $newEntry;

        DB::table('orders')->where('id', $order_id)->update([
            'rejected_by' => json_encode($existing),
            'updated_at' => Carbon::now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Order rejected successfully'
        ]);
    }


    public function getPendingOrders(Request $request)
    {
        $store = auth('store')->user();

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.rider_id',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name',
                'a.latitude',
                'a.longitude'
            )
            ->where('o.user_type', 'store')
            ->where('o.status', 'order_taken')
            ->where('o.with_shipping', 1)
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->orderByRaw("IF(o.payment_method = 'COD', o.created_at, o.paid_at) DESC")
            ->get();

            $orders = $orders->map(function ($order) {
            $dateTime = $order->payment_method === 'COD' ? $order->created_at : $order->paid_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'oi.id',
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category' => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand' => $item->brand,
                        'quantity' => $item->quantity,
                        'unit' => $item->quantity_unit,
                        'image' => $item->image ? asset($item->image) : null
                    ];
                });

            return [
                'order_id'   => $order->id,
                'order_no'   => $order->order_no,
                'status'     => $order->status,
                'shipping_type'  => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'rider_id'   => $order->rider_id,
                'rider_code' => $order->rider_code,

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'     => $order->door_no,
                'street_name' => $order->street_name,
                'area'        => $order->area,
                'city'        => $order->city,
                'district'    => $order->district_name,
                'state'       => $order->state_name,
                'pincode'     => $order->pincode,

                'order_items' => $items
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getAvailableRiders($order_id)
    {
        $store = auth('store')->user();
        $radiusKm = 10;

        if (!$store->latitude || !$store->longitude) {
            return response()->json([
                'success' => false,
                'message' => 'Store location not available'
            ], 400);
        }

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $lastRequestAt = DB::table('order_rider_requests')
            ->where('order_id', $order_id)
            ->max('requested_at');

        $enableSendNearby = true;
        $remainingMinutes = 0;

        if ($lastRequestAt) {
            $nextAllowedTime = Carbon::parse($lastRequestAt)->addHours(2);
            $now = now();

            if ($now->lessThan($nextAllowedTime)) {
                $enableSendNearby = false;
                $remainingMinutes = $now->diffInMinutes($nextAllowedTime);
            }
        }

        $riders = collect();

        if ($enableSendNearby) {
            $riders = DB::table('riders as r')
                ->join('rider_locations as rl', 'rl.rider_id', '=', 'r.id')
                ->select(
                    'r.id',
                    'r.rider_code',
                    'r.name',
                    'rl.latitude',
                    'rl.longitude',
                    DB::raw("(
                        6371 * acos(
                            cos(radians(?)) *
                            cos(radians(rl.latitude)) *
                            cos(radians(rl.longitude) - radians(?)) +
                            sin(radians(?)) *
                            sin(radians(rl.latitude))
                        )
                    ) AS distance")
                )
                ->addBinding($store->latitude)
                ->addBinding($store->longitude)
                ->addBinding($store->latitude)
                ->where('r.status', 'active')
                ->where('r.is_available', 1)
                ->whereNotNull('rl.latitude')
                ->whereNotNull('rl.longitude')
                ->having('distance', '<=', $radiusKm)
                ->orderBy('distance', 'asc')
                ->get();
        }

        return response()->json([
            'success' => true,
            'enable_send_nearby' => $enableSendNearby,
            'remaining_minutes' => $remainingMinutes,
            'count' => $riders->count(),
            'data'  => $riders
        ]);
    }


    public function sendRiderRequests(Request $request, $order_id)
    {
        $store = auth('store')->user();
        $radiusKm = 10;

        $request->validate([
            'rider_ids'   => 'required|array|min:1',
            'rider_ids.*' => 'exists:riders,id'
        ]);

        if (!$store->latitude || !$store->longitude) {
            return response()->json([
                'success' => false,
                'message' => 'Store location not available'
            ], 400);
        }

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        DB::beginTransaction();
        try {

            foreach ($request->rider_ids as $riderId) {

                $rider = DB::table('riders as r')
                    ->join('rider_locations as rl', 'rl.rider_id', '=', 'r.id')
                    ->select(
                     'r.id',
                        'r.is_available',
                        'rl.latitude',
                        'rl.longitude',
                        DB::raw("
                            (
                                6371 * acos(
                                    cos(radians(?)) *
                                    cos(radians(rl.latitude)) *
                                    cos(radians(rl.longitude) - radians(?)) +
                                    sin(radians(?)) *
                                    sin(radians(rl.latitude))
                                )
                            ) AS distance
                        ")
                    )
                    ->addBinding($store->latitude)
                    ->addBinding($store->longitude)
                    ->addBinding($store->latitude)
                    ->where('r.id', $riderId)
                    ->where('status', 'active')
                    ->where('is_available', 1)
                    // ->whereNotNull('latitude')
                    // ->whereNotNull('longitude')
                    ->first();

                if (!$rider || $rider->distance > $radiusKm) {
                    continue;
                }

                $existingRequest = DB::table('order_rider_requests')
                    ->where('order_id', $order_id)
                    ->where('rider_id', $riderId)
                    ->first();

                if ($existingRequest) {

                    $nextAllowedTime = Carbon::parse($existingRequest->requested_at)
                        ->addHours(2);

                    if (now()->lessThan($nextAllowedTime)) {
                        continue;
                    }

                    DB::table('order_rider_requests')
                        ->where('id', $existingRequest->id)
                        ->update([
                            'status'       => 'requested',
                            'requested_at' => now(),
                            'responded_at' => null,
                            'accepted_at'  => null,
                            'otp'          => null,
                            'updated_at'   => now(),
                        ]);

                } else {

                    DB::table('order_rider_requests')->insert([
                        'order_id'     => $order_id,
                        'rider_id'     => $riderId,
                        // 'warehouse_id' => $warehouse->id ?? null,
                        'store_id'     => $store_id->id ?? null,
                        'latitude'     => $rider->latitude,
                        'longitude'    => $rider->longitude,
                        'distance_km'  => round($rider->distance, 2),
                        'status'       => 'requested',
                        'requested_at' => now(),
                        'created_at'   => now(),
                        'updated_at'   => now(),
                    ]);
                }
            }

            // Update order status
            DB::table('orders')
                ->where('id', $order_id)
                ->update([
                    'status'     => 'waiting_for_rider_approval',
                    'updated_at' => now()
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pickup request sent successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function getAssignedOrders(Request $request)
    {
        $store = auth('store')->user();

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name'
            )
            ->where('o.user_type', 'store')
            ->whereIn('o.status', [
                'waiting_for_rider_approval',
                'rider_accepted'
            ])
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->orderByRaw("IF(o.payment_method = 'COD', o.created_at, o.paid_at) DESC")
            ->get();

        $orders = $orders->map(function ($order) {
            $dateTime = $order->payment_method === 'COD'
                ? $order->created_at
                : $order->paid_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                        'image'       => $item->image ? asset($item->image) : null
                    ];
                });

            return [
                'order_id' => $order->id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type'  => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'     => $order->door_no,
                'street_name' => $order->street_name,
                'area'        => $order->area,
                'city'        => $order->city,
                'district'    => $order->district_name,
                'state'       => $order->state_name,
                'pincode'     => $order->pincode,

                'rider' => $order->status === 'rider_accepted'
                    ? [
                        'name'   => $order->rider_name,
                        'code'   => $order->rider_code,
                        'mobile' => $order->rider_mobile,
                    ]
                    : null,

                'order_items' => $items
            ];
        });
        
        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getAwaitingRider($orderId)
    {
        $store = auth('store')->user();

        $order = DB::table('orders')
            ->where('id', $orderId)
            ->where('approved_by', $store->id)
            ->where('approved_by_user_type', 'store')
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $riderRequests = DB::table('order_rider_requests as orr')
        ->join('riders as r', 'r.id', '=', 'orr.rider_id')
        ->where('orr.order_id', $orderId)
        ->orderBy('orr.created_at', 'desc')
        ->select(
            'orr.id as request_id',
            'orr.status as request_status',
            'orr.requested_at',
            'orr.distance_km',
            'r.id as rider_id',
            'r.name',
            'r.rider_code',
        )
        ->get()
        ->map(function ($row) {
            return [
                'request_id'   => $row->request_id,
                'status'       => $row->request_status,
                'requested_at' => $row->requested_at,
                'distance_km'  => $row->distance_km,
                'rider' => [
                    'id'     => $row->rider_id,
                    'name'   => $row->name,
                    'code'   => $row->rider_code,
                ]
            ];
        });


        return response()->json([
            'success' => true,
            'order' => [
                'id'     => $order->id,
                'order_no' => $order->order_no,
                'status' => $order->status
            ],
            'count' => $riderRequests->count(),
            'data'  => $riderRequests
        ]);
    }

    public function canResendNearbyRiders($orderId)
    {
        $lastRequest = DB::table('order_rider_requests')
            ->where('order_id', $orderId)
            ->max('requested_at');

        if (!$lastRequest) {
            return response()->json([
                'success' => true,
                'enable_resend' => true,
                'remaining_minutes' => 0
            ]);
        }

        $nextAllowedTime = Carbon::parse($lastRequest)->addHours(2);
        $now = now();

        if ($now->greaterThanOrEqualTo($nextAllowedTime)) {
            return response()->json([
                'success' => true,
                'enable_resend' => true,
                'remaining_minutes' => 0
            ]);
        }

        return response()->json([
            'success' => true,
            'enable_resend' => false,
            'remaining_minutes' => $now->diffInMinutes($nextAllowedTime)
        ]);
    }

    public function getAssignedOrderDetail($orderId)
    {
        $store = auth('store')->user();

        $order = DB::table('orders as o')
            ->leftJoin('order_rider_requests as orr', function ($join) {
                $join->on('orr.order_id', '=', 'o.id')
                    ->where('orr.status', 'active');
            })
            ->leftJoin('riders as r', 'r.id', '=', 'orr.rider_id')
            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')
            ->leftJoin('stores as s', 's.id', '=', 'o.approved_by')
            ->leftJoin('districts as sd', 'sd.id', '=', 's.district_id')
            ->leftJoin('states as ss', 'ss.id', '=', 's.state_id')
            ->where('o.id', $orderId)
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.loaded_otp',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                's.store_name as pickup_name',
                's.door_no as pickup_door_no',
                's.street_name as pickup_street',
                's.area as pickup_area',
                's.city as pickup_city',
                'sd.name as pickup_district',
                'ss.name as pickup_state',
                's.pincode as pickup_pincode',
                's.latitude as pickup_latitude',
                's.longitude as pickup_longitude',
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'oi.quantity',
                'q.name as quantity_unit',
                'p.image'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'category'    => $item->category,
                    'subcategory' => $item->subcategory,
                    'brand'       => $item->brand,
                    'quantity'    => $item->quantity,
                    'unit'        => $item->quantity_unit,
                    'image'       => $item->image ? asset($item->image) : null
                ];
            });

        $dateTime = $order->payment_method === 'COD'
            ? $order->created_at
            : $order->paid_at;

         $expectedDeliveryDateTime = Carbon::parse($dateTime)->addDays(7);

        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                'date' => date('Y-m-d', strtotime($dateTime)),
                // 'time' => date('H:i:s', strtotime($dateTime)),
                'expected_delivery_date' => $expectedDeliveryDateTime->format('Y-m-d'),

                'products' => $products,

                'pickup_location' => [
                    'name'     => $order->pickup_name,
                    'door_no'  => $order->pickup_door_no,
                    'street'   => $order->pickup_street,
                    'area'     => $order->pickup_area,
                    'city'     => $order->pickup_city,
                    'district' => $order->pickup_district,
                    'state'    => $order->pickup_state,
                    'pincode'  => $order->pickup_pincode,
                ],

                'drop_location' => [
                    'name'     => $order->drop_name,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                ],

                'rider' => $order->rider_name ? [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ] : null,

                'loaded_otp' => $order->loaded_otp ?? null
            ]
        ]);
    }


    public function verifyRiderAndGenerateLoadOtp(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'otp'      => 'required|digits:4'
        ]);

        $store = auth('store')->user();
        $riderRequest = DB::table('order_rider_requests')
            ->where('order_id', $request->order_id)
            ->where('otp', $request->otp)
            ->where('status', 'active')
            ->first();

        if (!$riderRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid rider OTP'
            ], 400);
        }

        DB::beginTransaction();

        try {
            DB::table('order_rider_requests')
                ->where('id', $riderRequest->id)
                ->update([
                    'accepted_at'  => now(),
                    'responded_at' => now(),
                    'updated_at'   => now()
                ]);

            $loadedOtp = rand(1000, 9999);

            DB::table('orders')
                ->where('id', $request->order_id)
                ->update([
                    'status'     => 'rider_verified',
                    'loaded_otp' => $loadedOtp,
                    'loaded_at'  => now(),
                    'updated_at' => now()
                ]);
                
            DB::table('riders')
                ->where('id', $riderRequest->rider_id)
                ->update([
                    'work_status' => 0
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rider verified & loading OTP generated',
                'data' => [
                    'order_id'   => $request->order_id,
                    'loaded_otp' => $loadedOtp
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function getLoadedOrders(Request $request)
    {
        $store = auth('store')->user();

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.ride_started_at',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name'
            )
            ->where('o.user_type', 'store')
            ->where('o.status', 'in_transit')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->orderBy('o.ride_started_at', 'desc')
            ->get();

        $orders = $orders->map(function ($order) {

            // Use ride_started_at for loaded orders
            $dateTime = $order->ride_started_at ?? $order->paid_at ?? $order->created_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                        'image'       => $item->image ? asset($item->image) : null
                    ];
                });

            return [
                'order_id' => $order->id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'     => $order->door_no,
                'street_name' => $order->street_name,
                'area'        => $order->area,
                'city'        => $order->city,
                'district'    => $order->district_name,
                'state'       => $order->state_name,
                'pincode'     => $order->pincode,

                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],

                'order_items' => $items
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getLoadedOrderDetail($orderId)
    {
        $store = auth('store')->user();

        $order = DB::table('orders as o')

            /* Rider */
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')

            /* Drop location */
            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')

            /* Pickup location (store) */
            ->leftJoin('stores as s', 's.id', '=', 'o.approved_by')
            ->leftJoin('districts as sd', 'sd.id', '=', 's.district_id')
            ->leftJoin('states as ss', 'ss.id', '=', 's.state_id')

            /* Latest Rider Location (SUBQUERY JOIN) */
            ->leftJoin(DB::raw('
                (SELECT rl1.*
                FROM rider_locations rl1
                INNER JOIN (
                    SELECT rider_id, MAX(created_at) AS max_time
                    FROM rider_locations
                    GROUP BY rider_id
                ) rl2
                ON rl1.rider_id = rl2.rider_id 
                AND rl1.created_at = rl2.max_time
                ) as rl
            '), 'rl.rider_id', '=', 'o.rider_id')

            ->where('o.id', $orderId)
            ->where('o.status', 'in_transit')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.ride_started_at',

                /* Rider */
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',

                /* Rider live location */
                'rl.latitude as rider_latitude',
                'rl.longitude as rider_longitude',

                /* Pickup (store) */
                's.store_name as pickup_name',
                's.door_no as pickup_door_no',
                's.street_name as pickup_street',
                's.area as pickup_area',
                's.city as pickup_city',
                'sd.name as pickup_district',
                'ss.name as pickup_state',
                's.pincode as pickup_pincode',
                's.latitude as pickup_latitude',
                's.longitude as pickup_longitude',

                /* Drop */
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode',
                'da.latitude as drop_latitude',
                'da.longitude as drop_longitude'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Loaded order not found'
            ], 404);
        }

        /* ---------- Products / Materials ---------- */
        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'oi.quantity',
                'q.name as quantity_unit',
                'p.image'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'category'    => $item->category,
                    'subcategory' => $item->subcategory,
                    'brand'       => $item->brand,
                    'quantity'    => $item->quantity,
                    'unit'        => $item->quantity_unit,
                    'image'       => $item->image ? asset($item->image) : null
                ];
            });

        /* ---------- Base Date Logic ---------- */
        $baseDateTime = $order->payment_method === 'COD'
            ? $order->created_at
            : $order->paid_at;

        /* ---------- Expected Delivery Date (+7 days) ---------- */
        $expectedDeliveryDateTime = Carbon::parse($baseDateTime)->addDays(7);

        return response()->json([
            'success' => true,
            'data' => [

                /* Order */
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                /* Order Date */
                'order_date' => Carbon::parse($baseDateTime)->format('Y-m-d'),
                // 'order_time' => Carbon::parse($baseDateTime)->format('H:i:s'),

                /* Expected Delivery Date (+7 days) */
                'expected_delivery_date' => $expectedDeliveryDateTime->format('Y-m-d'),
                // 'expected_delivery_time' => $expectedDeliveryDateTime->format('H:i:s'),

                /* Products */
                'products' => $products,

                /* Pickup Location */
                // 'pickup_location' => [
                //     'name'     => $order->pickup_name,
                //     'door_no'  => $order->pickup_door_no,
                //     'street'   => $order->pickup_street,
                //     'area'     => $order->pickup_area,
                //     'city'     => $order->pickup_city,
                //     'district' => $order->pickup_district,
                //     'state'    => $order->pickup_state,
                //     'pincode'  => $order->pickup_pincode,
                //     'latitude' => $order->pickup_latitude,
                //     'longitude'=> $order->pickup_longitude,
                // ],

                /* Drop Location */
                'drop_location' => [
                    'name'     => $order->drop_name,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                    // 'latitude' => $order->drop_latitude,
                    // 'longitude'=> $order->drop_longitude,
                ],

                /* Rider */
                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],

                'tracking' => [
                    'rider_latitude'    => $order->rider_latitude,
                    'rider_longitude'   => $order->rider_longitude,
                    'pickup_latitude'   => $order->pickup_latitude,
                    'pickup_longitude'  => $order->pickup_longitude,
                    'drop_latitude'     => $order->drop_latitude,
                    'drop_longitude'    => $order->drop_longitude,
                ]
            ]
        ]);
    }

    public function getDeliveredOrders(Request $request)
    {
        $store = auth('store')->user();

        $query = DB::table('orders as o')

            /* Rider */
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')

            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')

            ->where('o.status', 'completed')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.delivered_at',

                /* Rider */
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',

                /* Drop */
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode'
            );

        if ($request->filled('date')) {
            $query->whereDate('o.delivered_at', $request->date);
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween(
                'o.delivered_at',
                [
                    Carbon::parse($request->from_date)->startOfDay(),
                    Carbon::parse($request->to_date)->endOfDay()
                ]
            );
        }

        if ($request->filled('month') && $request->filled('year')) {
            $query->whereMonth('o.delivered_at', $request->month)
                ->whereYear('o.delivered_at', $request->year);
        }

        if ($request->filled('year')) {
            $query->whereYear('o.delivered_at', $request->year);
        }

        $orders = $query->orderByDesc('o.delivered_at')->get();

        $orders = $orders->map(function ($order) {

            $deliveredDateTime = $order->delivered_at
                ?? $order->paid_at
                ?? $order->created_at;

            $products = DB::table('order_items as oi')
                ->join('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->order_id)
                ->select(
                    'c.name as category',
                    'sc.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                    ];
                });

            return [
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'delivered_date' => Carbon::parse($deliveredDateTime)->format('Y-m-d'),
                'delivered_time' => Carbon::parse($deliveredDateTime)->format('H:i:s'),
                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],

                'materials' => $products
            ];
        });

        return response()->json([
            'success' => true,
            'filters' => [
                'date'      => $request->date ?? null,
                'from_date' => $request->from_date ?? null,
                'to_date'   => $request->to_date ?? null,
                'month'     => $request->month ?? null,
                'year'      => $request->year ?? null,
            ],
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getDeliveredOrderDetail($orderId)
    {
        $store = auth('store')->user();

        $order = DB::table('orders as o')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')

            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')

            ->leftJoin('stores as w', 's.id', '=', 'o.approved_by')
            ->leftJoin('districts ss wd', 'sd.id', '=', 's.district_id')
            ->leftJoin('states as ss', 'ss.id', '=', 's.state_id')

            ->where('o.id', $orderId)
            ->where('o.status', 'completed')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.delivered_at',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                's.store_name as pickup_name',
                's.door_no as pickup_door_no',
                's.street_name as pickup_street',
                's.area as pickup_area',
                's.city as pickup_city',
                'sd.name as pickup_district',
                'ss.name as pickup_state',
                's.pincode as pickup_pincode',
                's.latitude as pickup_latitude',
                's.longitude as pickup_longitude',
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode',
                'da.latitude as drop_latitude',
                'da.longitude as drop_longitude'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Delivered order not found'
            ], 404);
        }

        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'oi.quantity',
                'q.name as quantity_unit',
                'p.image'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'category'    => $item->category,
                    'subcategory' => $item->subcategory,
                    'brand'       => $item->brand,
                    'quantity'    => $item->quantity,
                    'unit'        => $item->quantity_unit,
                    'image'       => $item->image ? asset($item->image) : null
                ];
            });

        $orderDateTime = $order->payment_method === 'COD'
            ? $order->created_at
            : $order->paid_at;

        $deliveredDateTime = $order->delivered_at ?? now();

        return response()->json([
            'success' => true,
            'data' => [

                /* Order Info */
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'payment_method' => $order->payment_method,
                'order_date' => Carbon::parse($orderDateTime)->format('Y-m-d'),
                'order_time' => Carbon::parse($orderDateTime)->format('H:i:s'),
                'delivered_date' => Carbon::parse($deliveredDateTime)->format('Y-m-d'),
                'delivered_time' => Carbon::parse($deliveredDateTime)->format('H:i:s'),
                'materials' => $products,
                'pickup_location' => [
                    'name'     => $order->pickup_name,
                    'door_no'  => $order->pickup_door_no,
                    'street'   => $order->pickup_street,
                    'area'     => $order->pickup_area,
                    'city'     => $order->pickup_city,
                    'district' => $order->pickup_district,
                    'state'    => $order->pickup_state,
                    'pincode'  => $order->pickup_pincode,
                    'latitude' => $order->pickup_latitude,
                    'longitude'=> $order->pickup_longitude,
                ],

                'drop_location' => [
                    'name'     => $order->drop_name,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                    'latitude' => $order->drop_latitude,
                    'longitude'=> $order->drop_longitude,
                ],

                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ]
            ]
        ]);
    }

    public function storeSalesReport(Request $request)
    {
        $store = auth('store')->user();

        $query = DB::table('orders as o')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->where('o.status', 'completed');

        if ($request->filled('date')) {
            $query->whereDate('o.delivered_at', $request->date);
        } elseif ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween('o.delivered_at', [
                $request->from_date . ' 00:00:00',
                $request->to_date . ' 23:59:59'
            ]);
        } elseif ($request->filled('month') && $request->filled('year')) {
            $query->whereMonth('o.delivered_at', $request->month)
                ->whereYear('o.delivered_at', $request->year);
        } elseif ($request->filled('year')) {
            $query->whereYear('o.delivered_at', $request->year);
        }

        $loadedQuery = DB::table('orders')
            ->where('approved_by', $store->id)
            ->where('approved_by_user_type', 'store')
            ->where('status', 'in_transit');

        if ($request->filled('date')) {
            $loadedQuery->whereDate('updated_at', $request->date); // use updated_at for in_transit
        } elseif ($request->filled('from_date') && $request->filled('to_date')) {
            $loadedQuery->whereBetween('updated_at', [
                $request->from_date . ' 00:00:00',
                $request->to_date . ' 23:59:59'
            ]);
        } elseif ($request->filled('month') && $request->filled('year')) {
            $loadedQuery->whereMonth('updated_at', $request->month)
                        ->whereYear('updated_at', $request->year);
        } elseif ($request->filled('year')) {
            $loadedQuery->whereYear('updated_at', $request->year);
        }

        $summary = (clone $query)
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->select(
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(oi.approved_unit_price * oi.quantity) as total_sale_amount'),
                DB::raw('0 as loaded_orders'), // placeholder
                DB::raw('COUNT(*) as delivered_orders'),
                DB::raw('SUM(CASE WHEN with_shipping = 1 THEN 1 ELSE 0 END) as with_shipping_orders'),
                DB::raw('SUM(CASE WHEN with_shipping = 0 THEN 1 ELSE 0 END) as without_shipping_orders')
            )
            ->first();

        // Count loaded orders separately
        $summary->loaded_orders = (clone $loadedQuery)->count();

        $categorySummary = DB::table('order_items as oi')
            ->join('orders as o', 'o.id', '=', 'oi.order_id')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->join('store_products as sp', function($join) use ($store) {
                $join->on('sp.product_id', '=', 'p.id')
                    ->where('sp.store_id', $store->id);
            })
            ->join('categories as c', 'c.id', '=', 'sp.category_id')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->where('o.status', 'completed')
            ->when($request->filled('date'), fn($q) => $q->whereDate('o.delivered_at', $request->date))
            ->when($request->filled('from_date') && $request->filled('to_date'), 
                fn($q) => $q->whereBetween('o.delivered_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']))
            ->when($request->filled('month') && $request->filled('year'), 
                fn($q) => $q->whereMonth('o.delivered_at', $request->month)->whereYear('o.delivered_at', $request->year))
            ->when($request->filled('year') && !$request->filled('month'), 
                fn($q) => $q->whereYear('o.delivered_at', $request->year))
            ->select(
                'c.id as category_id',
                'c.name as category_name',
                'c.image as category_image',
                'p.quantity_unit as unit',
                DB::raw('SUM(oi.quantity) as total_quantity'),
                DB::raw('SUM(oi.approved_unit_price * oi.quantity) as total_amount')
            )
            ->groupBy('c.id', 'p.quantity_unit')
            ->orderBy('c.name')
            ->get()
            ->map(function ($item) {
                return [
                    'category_id'    => $item->category_id,
                    'category_name'  => $item->category_name,
                    'category_image' => $item->category_image ? asset($item->category_image) : null,
                    'unit'           => $item->unit,
                    'total_quantity' => $item->total_quantity
                ];
            });

        $paymentSummary = (clone $query)
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->select(
                DB::raw("SUM(CASE WHEN o.payment_method = 'COD' THEN (oi.approved_unit_price * oi.quantity) ELSE 0 END) as cod_amount"),
                DB::raw("SUM(CASE WHEN o.payment_method IN ('Razorpay','Cashfree') THEN (oi.approved_unit_price * oi.quantity) ELSE 0 END) as online_amount")
            )
            ->first();

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $filterContent = 'Sales Report from ' . date('d-M-Y', strtotime($request->from_date))
                        . ' to ' . date('d-M-Y', strtotime($request->to_date));
        } elseif ($request->filled('date')) {
            $filterContent = 'Sales Report for ' . date('d-M-Y', strtotime($request->date));
        } elseif ($request->filled('month') && $request->filled('year')) {
            $filterContent = 'Sales Report for ' . date('F', mktime(0,0,0,$request->month,1)) . ' ' . $request->year;
        } elseif ($request->filled('year')) {
            $filterContent = 'Sales Report for Year ' . $request->year;
        } else {
            $filterContent = 'Sales Report';
        }

        return response()->json([
            'success' => true,
            'filter_content' => $filterContent,
            'summary' => [
                'total_sale_amount'        => (float) ($summary->total_sale_amount ?? 0),
                'total_orders'             => (int) ($summary->total_orders ?? 0),
                'loaded_orders'            => (int) ($summary->loaded_orders ?? 0),
                'delivered_orders'         => (int) ($summary->delivered_orders ?? 0),
                'with_shipping_orders'     => (int) ($summary->with_shipping_orders ?? 0),
                'without_shipping_orders'  => (int) ($summary->without_shipping_orders ?? 0)
            ],
            'category_summary' => $categorySummary,
            'payment_summary' => [
                'cod_amount'    => (float) ($paymentSummary->cod_amount ?? 0),
                'online_amount' => (float) ($paymentSummary->online_amount ?? 0),
            ]
        ]);
    }

    public function storeSalesDashboard(Request $request)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $today = Carbon::today();

        // Last week (Monday → Sunday)
        $lastWeekStart = Carbon::now()->subWeek()->startOfWeek(Carbon::MONDAY);
        $lastWeekEnd   = Carbon::now()->subWeek()->endOfWeek(Carbon::SUNDAY);

        $todayOrderTakenCount = DB::table('orders')
            ->where('approved_by', $store->id)
            ->where('approved_by_user_type', 'store')
            ->whereDate('approved_at', $today)
            ->count();

        $todayLoadedOrderCount = DB::table('orders')
            ->where('approved_by', $store->id)
            ->where('approved_by_user_type', 'store')
            ->whereDate('loaded_at', $today)
            ->count();

        $lastWeekRevenue = DB::table('orders as o')
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->where('o.status', 'completed')
            ->whereBetween('o.approved_at', [$lastWeekStart, $lastWeekEnd])
            ->select(DB::raw('SUM(oi.approved_unit_price * oi.quantity) as revenue'))
            ->value('revenue');

        $highestCategory = DB::table('orders as o')
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->join('store_products as sp', function ($join) use ($store) {
                $join->on('sp.product_id', '=', 'oi.product_id')
                    ->where('sp.store_id', '=', $store->id);
            })
            ->select('sp.category_id', DB::raw('SUM(oi.total_price) as total_amount'))
            ->where('o.approved_by', $store->id)
            ->where('o.approved_by_user_type', 'store')
            ->where('o.status', 'completed')
            ->whereBetween('o.approved_at', values: [$lastWeekStart, $lastWeekEnd])
            ->groupBy('sp.category_id')
            ->orderByDesc('total_amount')
            ->first();

        $highestCategoryData = null;

        if ($highestCategory && $highestCategory->category_id) {
            $category = DB::table('categories')
                ->where('id', $highestCategory->category_id)
                ->first();

            $highestCategoryData = [
                'category_id'   => $highestCategory->category_id,
                'category_name' => $category->name ?? null,
                'amount'        => round($highestCategory->total_amount, 2)
            ];
        }

        $pendingOrdersCount = DB::table('orders')
            ->where('approved_by', $store->id)
            ->where('approved_by_user_type', 'store')
            ->where('status', 'order_taken')
            ->count();

        return response()->json([
            'success' => true,
            'store_id' => $store->id,
            'data' => [
                'today' => [
                    'order_taken_count'  => $todayOrderTakenCount,
                    'loaded_order_count' => $todayLoadedOrderCount,
                ],

                'last_week' => [
                    'from' => $lastWeekStart->format('Y-m-d'),
                    'to'   => $lastWeekEnd->format('Y-m-d'),
                    'total_revenue' => round($lastWeekRevenue, 2),
                    'highest_category' => $highestCategoryData
                ],

                'pending_orders_count' => $pendingOrdersCount
            ]
        ]);
    }

}
